/**
 * Player de Vídeo - WebPlayer
 */

const Player = {
    video: null,
    currentContent: null,
    isPlaying: false,
    
    /**
     * Inicializar player
     */
    init() {
        this.video = document.getElementById('video-player');
        
        if (this.video) {
            this.setupEventListeners();
        }
    },
    
    /**
     * Configurar event listeners
     */
    setupEventListeners() {
        // Eventos do vídeo
        this.video.addEventListener('play', () => {
            this.isPlaying = true;
        });
        
        this.video.addEventListener('pause', () => {
            this.isPlaying = false;
        });
        
        this.video.addEventListener('ended', () => {
            this.onVideoEnded();
        });
        
        this.video.addEventListener('error', (e) => {
            this.handleError(e);
        });
        
        this.video.addEventListener('timeupdate', () => {
            this.onTimeUpdate();
        });
        
        // Teclas de atalho
        document.addEventListener('keydown', (e) => {
            if (App.currentScreen === 'player') {
                this.handleKeyPress(e);
            }
        });
    },
    
    /**
     * Carregar e reproduzir conteúdo
     */
    async play(content) {
        this.currentContent = content;
        
        // Mostrar tela do player
        App.showScreen('player');
        
        // Atualizar informações
        document.getElementById('player-title').textContent = content.nome || content.titulo;
        document.getElementById('player-description').textContent = content.descricao || '';
        
        // Carregar stream
        const streamUrl = this.getStreamUrl(content);
        
        if (!streamUrl) {
            alert('URL de stream não disponível');
            this.close();
            return;
        }
        
        this.video.src = streamUrl;
        
        // Configurar volume
        this.video.volume = CONFIG.VIDEO.DEFAULT_VOLUME;
        
        // Reproduzir
        try {
            await this.video.play();
            
            // Adicionar ao histórico
            this.addToHistory();
        } catch (error) {
            console.error('Erro ao reproduzir:', error);
            alert('Erro ao reproduzir o vídeo');
        }
    },
    
    /**
     * Obter URL do stream
     */
    getStreamUrl(content) {
        // Para canais ao vivo
        if (content.stream_url) {
            return content.stream_url;
        }
        
        // Para filmes/séries
        if (content.url) {
            return content.url;
        }
        
        return null;
    },
    
    /**
     * Toggle play/pause
     */
    togglePlay() {
        if (this.isPlaying) {
            this.video.pause();
        } else {
            this.video.play();
        }
    },
    
    /**
     * Parar
     */
    stop() {
        this.video.pause();
        this.video.currentTime = 0;
        this.close();
    },
    
    /**
     * Fechar player
     */
    close() {
        this.video.pause();
        this.video.src = '';
        this.currentContent = null;
        App.showScreen('main');
    },
    
    /**
     * Avançar
     */
    forward(seconds = 10) {
        this.video.currentTime += seconds;
    },
    
    /**
     * Retroceder
     */
    rewind(seconds = 10) {
        this.video.currentTime -= seconds;
    },
    
    /**
     * Ajustar volume
     */
    setVolume(volume) {
        this.video.volume = Math.max(0, Math.min(1, volume));
    },
    
    /**
     * Aumentar volume
     */
    volumeUp() {
        this.setVolume(this.video.volume + 0.1);
    },
    
    /**
     * Diminuir volume
     */
    volumeDown() {
        this.setVolume(this.video.volume - 0.1);
    },
    
    /**
     * Mute/Unmute
     */
    toggleMute() {
        this.video.muted = !this.video.muted;
    },
    
    /**
     * Fullscreen
     */
    toggleFullscreen() {
        if (!document.fullscreenElement) {
            if (this.video.requestFullscreen) {
                this.video.requestFullscreen();
            } else if (this.video.webkitRequestFullscreen) {
                this.video.webkitRequestFullscreen();
            } else if (this.video.msRequestFullscreen) {
                this.video.msRequestFullscreen();
            }
        } else {
            if (document.exitFullscreen) {
                document.exitFullscreen();
            } else if (document.webkitExitFullscreen) {
                document.webkitExitFullscreen();
            } else if (document.msExitFullscreen) {
                document.msExitFullscreen();
            }
        }
    },
    
    /**
     * Quando o vídeo termina
     */
    onVideoEnded() {
        // Salvar posição final no histórico
        this.savePosition();
        
        // Se for série, pode avançar para próximo episódio
        if (this.currentContent.tipo === 'serie' && this.currentContent.proximo_episodio) {
            // Mostrar opção de próximo episódio
            this.showNextEpisode();
        }
    },
    
    /**
     * Atualização de tempo
     */
    onTimeUpdate() {
        // Salvar posição periodicamente (a cada 30 segundos)
        if (Math.floor(this.video.currentTime) % 30 === 0) {
            this.savePosition();
        }
    },
    
    /**
     * Salvar posição no histórico
     */
    async savePosition() {
        if (!this.currentContent) return;
        
        const posicao = Math.floor(this.video.currentTime);
        
        await API.addHistorico(
            this.currentContent.tipo,
            this.currentContent.id,
            posicao
        );
    },
    
    /**
     * Adicionar ao histórico
     */
    async addToHistory() {
        if (!this.currentContent) return;
        
        await API.addHistorico(
            this.currentContent.tipo,
            this.currentContent.id,
            0
        );
    },
    
    /**
     * Tratar erros
     */
    handleError(error) {
        console.error('Erro no player:', error);
        
        let message = 'Erro ao reproduzir o vídeo';
        
        if (this.video.error) {
            switch (this.video.error.code) {
                case 1:
                    message = 'Reprodução abortada';
                    break;
                case 2:
                    message = 'Erro de rede';
                    break;
                case 3:
                    message = 'Erro ao decodificar';
                    break;
                case 4:
                    message = 'Formato não suportado';
                    break;
            }
        }
        
        alert(message);
        this.close();
    },
    
    /**
     * Teclas de atalho
     */
    handleKeyPress(e) {
        switch (e.key) {
            case ' ':
            case 'k':
                e.preventDefault();
                this.togglePlay();
                break;
            case 'f':
                e.preventDefault();
                this.toggleFullscreen();
                break;
            case 'm':
                e.preventDefault();
                this.toggleMute();
                break;
            case 'ArrowLeft':
                e.preventDefault();
                this.rewind();
                break;
            case 'ArrowRight':
                e.preventDefault();
                this.forward();
                break;
            case 'ArrowUp':
                e.preventDefault();
                this.volumeUp();
                break;
            case 'ArrowDown':
                e.preventDefault();
                this.volumeDown();
                break;
            case 'Escape':
                e.preventDefault();
                this.close();
                break;
        }
    },
    
    /**
     * Mostrar próximo episódio
     */
    showNextEpisode() {
        // Implementar lógica de próximo episódio
        // Por enquanto, apenas fecha o player
        setTimeout(() => {
            this.close();
        }, 3000);
    }
};
