/**
 * Gerenciamento de Autenticação - WebPlayer
 */

const Auth = {
    /**
     * Fazer login
     */
    async login(numeroProvedor, usuario, senha) {
        try {
            const result = await API.login(numeroProvedor, usuario, senha);
            
            if (result && result.success) {
                this.setAuth(result.data);
                return { success: true };
            } else {
                return { 
                    success: false, 
                    message: result ? result.message : 'Erro ao fazer login' 
                };
            }
        } catch (error) {
            return { success: false, message: 'Erro ao conectar com o servidor' };
        }
    },
    
    /**
     * Salvar dados de autenticação
     */
    setAuth(data) {
        if (data.token) {
            localStorage.setItem(CONFIG.STORAGE_KEYS.AUTH_TOKEN, data.token);
        }
        
        if (data.refresh_token) {
            localStorage.setItem(CONFIG.STORAGE_KEYS.REFRESH_TOKEN, data.refresh_token);
        }
        
        if (data.usuario) {
            localStorage.setItem(CONFIG.STORAGE_KEYS.USER_DATA, JSON.stringify(data.usuario));
        }
    },
    
    /**
     * Obter token
     */
    getToken() {
        return localStorage.getItem(CONFIG.STORAGE_KEYS.AUTH_TOKEN);
    },
    
    /**
     * Obter refresh token
     */
    getRefreshToken() {
        return localStorage.getItem(CONFIG.STORAGE_KEYS.REFRESH_TOKEN);
    },
    
    /**
     * Obter dados do usuário
     */
    getUser() {
        const userData = localStorage.getItem(CONFIG.STORAGE_KEYS.USER_DATA);
        return userData ? JSON.parse(userData) : null;
    },
    
    /**
     * Verificar se está autenticado
     */
    isAuthenticated() {
        return !!this.getToken();
    },
    
    /**
     * Limpar autenticação
     */
    clearAuth() {
        localStorage.removeItem(CONFIG.STORAGE_KEYS.AUTH_TOKEN);
        localStorage.removeItem(CONFIG.STORAGE_KEYS.REFRESH_TOKEN);
        localStorage.removeItem(CONFIG.STORAGE_KEYS.USER_DATA);
        localStorage.removeItem(CONFIG.STORAGE_KEYS.CURRENT_PROFILE);
    },
    
    /**
     * Fazer logout
     */
    logout() {
        this.clearAuth();
        App.showLogin();
    },
    
    /**
     * Renovar token
     */
    async refreshToken() {
        const refreshToken = this.getRefreshToken();
        
        if (!refreshToken) {
            this.logout();
            return false;
        }
        
        try {
            const result = await API.refreshToken(refreshToken);
            
            if (result && result.success) {
                this.setAuth(result.data);
                return true;
            } else {
                this.logout();
                return false;
            }
        } catch (error) {
            this.logout();
            return false;
        }
    },
    
    /**
     * Verificar sessão
     */
    async checkSession() {
        if (!this.isAuthenticated()) {
            return false;
        }
        
        // Tentar renovar token se estiver próximo de expirar
        // (implementar lógica de verificação de expiração se necessário)
        
        return true;
    },
    
    /**
     * Perfil atual
     */
    getCurrentProfile() {
        const profileData = localStorage.getItem(CONFIG.STORAGE_KEYS.CURRENT_PROFILE);
        return profileData ? JSON.parse(profileData) : null;
    },
    
    /**
     * Definir perfil atual
     */
    setCurrentProfile(profile) {
        localStorage.setItem(CONFIG.STORAGE_KEYS.CURRENT_PROFILE, JSON.stringify(profile));
    },
    
    /**
     * Limpar perfil atual
     */
    clearCurrentProfile() {
        localStorage.removeItem(CONFIG.STORAGE_KEYS.CURRENT_PROFILE);
    }
};
