/**
 * Aplicação Principal - WebPlayer
 */

const App = {
    currentScreen: 'login',
    currentPage: 'home',
    profiles: [],
    currentProfile: null,
    
    /**
     * Inicializar aplicação
     */
    init() {
        console.log('Neuro Play IPTV - WebPlayer v' + CONFIG.VERSION);
        
        // Inicializar player
        Player.init();
        
        // Verificar autenticação
        if (Auth.isAuthenticated()) {
            this.showProfiles();
        } else {
            this.showLogin();
        }
        
        // Event listeners
        this.setupEventListeners();
        
        // Scroll header
        this.setupScrollHeader();
    },
    
    /**
     * Configurar event listeners
     */
    setupEventListeners() {
        // Formulário de login
        const loginForm = document.getElementById('login-form');
        if (loginForm) {
            loginForm.addEventListener('submit', (e) => this.handleLogin(e));
        }
        
        // Navegação
        const navItems = document.querySelectorAll('.nav-item');
        navItems.forEach(item => {
            item.addEventListener('click', (e) => {
                e.preventDefault();
                const page = item.dataset.page;
                this.loadPage(page);
            });
        });
    },
    
    /**
     * Configurar header scroll
     */
    setupScrollHeader() {
        window.addEventListener('scroll', () => {
            const header = document.querySelector('.main-header');
            if (header) {
                if (window.scrollY > 50) {
                    header.classList.add('scrolled');
                } else {
                    header.classList.remove('scrolled');
                }
            }
        });
    },
    
    /**
     * Mostrar tela
     */
    showScreen(screenName) {
        // Esconder todas as telas
        document.querySelectorAll('.screen').forEach(screen => {
            screen.classList.remove('active');
        });
        
        // Mostrar tela solicitada
        const screen = document.getElementById(`${screenName}-screen`);
        if (screen) {
            screen.classList.add('active');
            this.currentScreen = screenName;
        }
    },
    
    /**
     * Mostrar login
     */
    showLogin() {
        this.showScreen('login');
        document.getElementById('numero-provedor').focus();
    },
    
    /**
     * Handle login
     */
    async handleLogin(e) {
        e.preventDefault();
        
        const numeroProvedor = document.getElementById('numero-provedor').value;
        const usuario = document.getElementById('usuario').value;
        const senha = document.getElementById('senha').value;
        const errorDiv = document.getElementById('login-error');
        
        errorDiv.textContent = '';
        
        const result = await Auth.login(numeroProvedor, usuario, senha);
        
        if (result.success) {
            this.showProfiles();
        } else {
            errorDiv.textContent = result.message || 'Erro ao fazer login';
        }
    },
    
    /**
     * Mostrar perfis
     */
    async showProfiles() {
        this.showScreen('profile');
        
        // Carregar perfis
        const result = await API.getPerfis();
        
        if (result && result.success) {
            this.profiles = result.data || [];
            this.renderProfiles();
        } else {
            alert('Erro ao carregar perfis');
            this.logout();
        }
    },
    
    /**
     * Renderizar perfis
     */
    renderProfiles() {
        const grid = document.getElementById('profiles-grid');
        
        if (this.profiles.length === 0) {
            grid.innerHTML = '<p>Nenhum perfil encontrado. Crie um novo perfil.</p>';
            return;
        }
        
        grid.innerHTML = this.profiles.map(profile => `
            <div class="profile-card" onclick="App.selectProfile(${profile.id})">
                <div class="profile-avatar">
                    ${profile.avatar || '👤'}
                </div>
                <div class="profile-name">${profile.nome}</div>
            </div>
        `).join('');
        
        // Adicionar botão de criar perfil se não atingiu o limite
        if (this.profiles.length < CONFIG.LIMITS.MAX_PROFILES) {
            grid.innerHTML += `
                <div class="profile-card" onclick="App.createProfile()">
                    <div class="profile-avatar">➕</div>
                    <div class="profile-name">Novo Perfil</div>
                </div>
            `;
        }
    },
    
    /**
     * Selecionar perfil
     */
    async selectProfile(profileId) {
        const profile = this.profiles.find(p => p.id === profileId);
        
        if (!profile) return;
        
        // Verificar controle parental
        if (profile.pin_parental) {
            const pin = prompt('Digite o PIN parental:');
            if (pin !== profile.pin_parental) {
                alert('PIN incorreto');
                return;
            }
        }
        
        this.currentProfile = profile;
        Auth.setCurrentProfile(profile);
        
        // Atualizar nome do perfil no header
        document.getElementById('current-profile-name').textContent = profile.nome;
        
        // Ir para tela principal
        this.showHome();
    },
    
    /**
     * Criar perfil
     */
    async createProfile() {
        const nome = prompt('Nome do perfil:');
        if (!nome) return;
        
        const avatar = prompt('Avatar (emoji):') || '👤';
        
        const result = await API.createPerfil({ nome, avatar });
        
        if (result && result.success) {
            alert('Perfil criado com sucesso!');
            this.showProfiles();
        } else {
            alert('Erro ao criar perfil');
        }
    },
    
    /**
     * Mostrar tela principal
     */
    showHome() {
        this.showScreen('main');
        this.loadPage('home');
    },
    
    /**
     * Carregar página
     */
    async loadPage(page) {
        this.currentPage = page;
        
        // Atualizar navegação ativa
        document.querySelectorAll('.nav-item').forEach(item => {
            item.classList.remove('active');
            if (item.dataset.page === page) {
                item.classList.add('active');
            }
        });
        
        const content = document.getElementById('main-content');
        content.innerHTML = '<div class="loading"><div class="spinner"></div><p>Carregando...</p></div>';
        
        switch (page) {
            case 'home':
                await this.loadHome();
                break;
            case 'live':
                await this.loadLive();
                break;
            case 'movies':
                await this.loadMovies();
                break;
            case 'series':
                await this.loadSeries();
                break;
            case 'favorites':
                await this.loadFavorites();
                break;
        }
    },
    
    /**
     * Carregar página inicial
     */
    async loadHome() {
        const content = document.getElementById('main-content');
        
        // Carregar conteúdo em destaque
        const canais = await API.getCanais();
        const filmes = await API.getFilmes();
        const series = await API.getSeries();
        
        const canaisData = (canais && canais.success) ? canais.data : [];
        const filmesData = (filmes && filmes.success) ? filmes.data : [];
        const seriesData = (series && series.success) ? series.data : [];
        
        // Banner hero (primeiro canal ou filme)
        const destaque = canaisData[0] || filmesData[0] || seriesData[0];
        
        content.innerHTML = `
            ${destaque ? `
                <div class="hero-banner" style="background-image: linear-gradient(to right, rgba(0,0,0,0.8), transparent), url('${destaque.imagem || ''}')">
                    <div class="hero-content">
                        <h1 class="hero-title">${destaque.nome || destaque.titulo}</h1>
                        <p class="hero-description">${destaque.descricao || 'Sem descrição'}</p>
                        <div class="hero-buttons">
                            <button class="btn-hero btn-play" onclick="Player.play(${JSON.stringify(destaque).replace(/"/g, '&quot;')})">
                                ▶️ Assistir
                            </button>
                            <button class="btn-hero btn-info" onclick="App.showInfo(${JSON.stringify(destaque).replace(/"/g, '&quot;')})">
                                ℹ️ Mais Informações
                            </button>
                        </div>
                    </div>
                </div>
            ` : ''}
            
            ${this.renderContentSection('Canais ao Vivo', canaisData)}
            ${this.renderContentSection('Filmes', filmesData)}
            ${this.renderContentSection('Séries', seriesData)}
        `;
    },
    
    /**
     * Carregar canais ao vivo
     */
    async loadLive() {
        const content = document.getElementById('main-content');
        const result = await API.getCanais();
        const canais = (result && result.success) ? result.data : [];
        
        content.innerHTML = `
            <div class="content-section">
                <h2 class="section-title">Canais ao Vivo</h2>
                ${this.renderContentGrid(canais)}
            </div>
        `;
    },
    
    /**
     * Carregar filmes
     */
    async loadMovies() {
        const content = document.getElementById('main-content');
        const result = await API.getFilmes();
        const filmes = (result && result.success) ? result.data : [];
        
        content.innerHTML = `
            <div class="content-section">
                <h2 class="section-title">Filmes</h2>
                ${this.renderContentGrid(filmes)}
            </div>
        `;
    },
    
    /**
     * Carregar séries
     */
    async loadSeries() {
        const content = document.getElementById('main-content');
        const result = await API.getSeries();
        const series = (result && result.success) ? result.data : [];
        
        content.innerHTML = `
            <div class="content-section">
                <h2 class="section-title">Séries</h2>
                ${this.renderContentGrid(series)}
            </div>
        `;
    },
    
    /**
     * Carregar favoritos
     */
    async loadFavorites() {
        const content = document.getElementById('main-content');
        const result = await API.getFavoritos();
        const favoritos = (result && result.success) ? result.data : [];
        
        content.innerHTML = `
            <div class="content-section">
                <h2 class="section-title">Meus Favoritos</h2>
                ${favoritos.length > 0 ? this.renderContentGrid(favoritos) : '<p>Nenhum favorito adicionado</p>'}
            </div>
        `;
    },
    
    /**
     * Renderizar seção de conteúdo
     */
    renderContentSection(title, items) {
        if (!items || items.length === 0) return '';
        
        return `
            <div class="content-section">
                <h2 class="section-title">${title}</h2>
                <div class="content-row">
                    ${items.slice(0, 10).map(item => this.renderContentCard(item)).join('')}
                </div>
            </div>
        `;
    },
    
    /**
     * Renderizar grid de conteúdo
     */
    renderContentGrid(items) {
        if (!items || items.length === 0) {
            return '<p>Nenhum conteúdo disponível</p>';
        }
        
        return `
            <div class="content-row" style="flex-wrap: wrap;">
                ${items.map(item => this.renderContentCard(item)).join('')}
            </div>
        `;
    },
    
    /**
     * Renderizar card de conteúdo
     */
    renderContentCard(item) {
        const itemJson = JSON.stringify(item).replace(/"/g, '&quot;');
        
        return `
            <div class="content-card" onclick="Player.play(${itemJson})">
                <img src="${item.imagem || 'assets/images/placeholder.jpg'}" alt="${item.nome || item.titulo}" onerror="this.src='assets/images/placeholder.jpg'">
                <div class="content-card-title">${item.nome || item.titulo}</div>
            </div>
        `;
    },
    
    /**
     * Mostrar busca
     */
    showSearch() {
        const modal = document.getElementById('search-modal');
        modal.classList.add('active');
        document.getElementById('search-input').focus();
        
        // Event listener para busca
        const searchInput = document.getElementById('search-input');
        searchInput.addEventListener('input', (e) => this.handleSearch(e.target.value));
    },
    
    /**
     * Fechar busca
     */
    closeSearch() {
        const modal = document.getElementById('search-modal');
        modal.classList.remove('active');
        document.getElementById('search-input').value = '';
        document.getElementById('search-results').innerHTML = '';
    },
    
    /**
     * Handle busca
     */
    async handleSearch(query) {
        if (query.length < 3) {
            document.getElementById('search-results').innerHTML = '';
            return;
        }
        
        const result = await API.buscar(query);
        const results = (result && result.success) ? result.data : [];
        
        const resultsDiv = document.getElementById('search-results');
        
        if (results.length === 0) {
            resultsDiv.innerHTML = '<p>Nenhum resultado encontrado</p>';
        } else {
            resultsDiv.innerHTML = results.map(item => this.renderContentCard(item)).join('');
        }
    },
    
    /**
     * Mostrar menu de perfil
     */
    showProfileMenu() {
        const modal = document.getElementById('profile-menu-modal');
        modal.classList.add('active');
    },
    
    /**
     * Fechar menu de perfil
     */
    closeProfileMenu() {
        const modal = document.getElementById('profile-menu-modal');
        modal.classList.remove('active');
    },
    
    /**
     * Mostrar informações
     */
    showInfo(item) {
        alert(`${item.nome || item.titulo}\n\n${item.descricao || 'Sem descrição'}`);
    },
    
    /**
     * Logout
     */
    logout() {
        if (confirm('Deseja realmente sair?')) {
            Auth.logout();
        }
    }
};

// Inicializar quando DOM carregar
document.addEventListener('DOMContentLoaded', () => {
    App.init();
});
