/**
 * Cliente da API - WebPlayer
 */

const API = {
    /**
     * Requisição base
     */
    async request(endpoint, options = {}) {
        const token = Auth.getToken();
        
        const defaultOptions = {
            headers: {
                'Content-Type': 'application/json'
            }
        };
        
        if (token) {
            defaultOptions.headers['Authorization'] = 'Bearer ' + token;
        }
        
        const finalOptions = {
            ...defaultOptions,
            ...options,
            headers: {
                ...defaultOptions.headers,
                ...options.headers
            }
        };
        
        try {
            const response = await fetch(CONFIG.API_URL + endpoint, finalOptions);
            const text = await response.text();
            
            if (!text || text.trim() === '') {
                return { success: false, message: 'Resposta vazia do servidor' };
            }
            
            let data;
            try {
                data = JSON.parse(text);
            } catch (e) {
                console.error('Erro ao parsear JSON:', text);
                return { success: false, message: 'Resposta inválida do servidor' };
            }
            
            if (response.status === 401) {
                Auth.clearAuth();
                App.showLogin();
                return null;
            }
            
            return data;
        } catch (error) {
            console.error('Erro na requisição:', error);
            return { success: false, message: 'Erro ao conectar com o servidor' };
        }
    },
    
    /**
     * GET
     */
    async get(endpoint) {
        return this.request(endpoint, { method: 'GET' });
    },
    
    /**
     * POST
     */
    async post(endpoint, data) {
        return this.request(endpoint, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    },
    
    /**
     * PUT
     */
    async put(endpoint, data) {
        return this.request(endpoint, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    },
    
    /**
     * DELETE
     */
    async delete(endpoint) {
        return this.request(endpoint, { method: 'DELETE' });
    },
    
    // ==================== ENDPOINTS ESPECÍFICOS ====================
    
    /**
     * Login
     */
    async login(numeroProvedor, usuario, senha) {
        return this.post('auth/login', {
            numero_provedor: numeroProvedor,
            usuario: usuario,
            senha: senha,
            tipo: 'usuario'
        });
    },
    
    /**
     * Refresh Token
     */
    async refreshToken(refreshToken) {
        return this.post('auth/refresh', { refresh_token: refreshToken });
    },
    
    /**
     * Perfis
     */
    async getPerfis() {
        return this.get('usuario/perfis');
    },
    
    async createPerfil(data) {
        return this.post('usuario/perfis', data);
    },
    
    async updatePerfil(id, data) {
        return this.put(`usuario/perfis/${id}`, data);
    },
    
    async deletePerfil(id) {
        return this.delete(`usuario/perfis/${id}`);
    },
    
    /**
     * Conteúdo
     */
    async getCanais(categoria = null) {
        const endpoint = categoria ? `usuario/canais?categoria=${categoria}` : 'usuario/canais';
        return this.get(endpoint);
    },
    
    async getFilmes(categoria = null) {
        const endpoint = categoria ? `usuario/filmes?categoria=${categoria}` : 'usuario/filmes';
        return this.get(endpoint);
    },
    
    async getSeries(categoria = null) {
        const endpoint = categoria ? `usuario/series?categoria=${categoria}` : 'usuario/series';
        return this.get(endpoint);
    },
    
    async getConteudoDetalhes(tipo, id) {
        return this.get(`usuario/${tipo}/${id}`);
    },
    
    /**
     * Categorias
     */
    async getCategorias(tipo) {
        return this.get(`usuario/categorias?tipo=${tipo}`);
    },
    
    /**
     * Favoritos
     */
    async getFavoritos() {
        return this.get('usuario/favoritos');
    },
    
    async addFavorito(tipo, conteudoId) {
        return this.post('usuario/favoritos', {
            tipo: tipo,
            conteudo_id: conteudoId
        });
    },
    
    async removeFavorito(id) {
        return this.delete(`usuario/favoritos/${id}`);
    },
    
    /**
     * Histórico
     */
    async getHistorico() {
        return this.get('usuario/historico');
    },
    
    async addHistorico(tipo, conteudoId, posicao = 0) {
        return this.post('usuario/historico', {
            tipo: tipo,
            conteudo_id: conteudoId,
            posicao: posicao
        });
    },
    
    /**
     * Busca
     */
    async buscar(query) {
        return this.get(`usuario/buscar?q=${encodeURIComponent(query)}`);
    },
    
    /**
     * Mensagens e Novidades
     */
    async getMensagens() {
        return this.get('usuario/mensagens');
    },
    
    async getNovidades() {
        return this.get('usuario/novidades');
    },
    
    /**
     * Provedor Info
     */
    async getProvedorInfo() {
        return this.get('usuario/provedor');
    }
};
