<?php
/**
 * Parser de Playlists M3U
 */
class M3UParser {
    
    public static function parse($url) {
        try {
            $content = self::fetchContent($url);
            
            if (empty($content)) {
                throw new Exception('Playlist vazia ou inacessível');
            }
            
            $lines = explode("\n", $content);
            $channels = [];
            $currentChannel = null;
            
            foreach ($lines as $line) {
                $line = trim($line);
                
                if (strpos($line, '#EXTINF:') === 0) {
                    $currentChannel = self::parseExtinf($line);
                } elseif (!empty($line) && strpos($line, '#') !== 0 && $currentChannel) {
                    $currentChannel['url'] = $line;
                    $channels[] = $currentChannel;
                    $currentChannel = null;
                }
            }
            
            return $channels;
        } catch (Exception $e) {
            throw new Exception('Erro ao processar playlist: ' . $e->getMessage());
        }
    }
    
    private static function parseExtinf($line) {
        $channel = [
            'id' => null,
            'name' => '',
            'logo' => null,
            'group' => null,
            'url' => null
        ];
        
        // Extrair nome do canal
        if (preg_match('/,(.+)$/', $line, $matches)) {
            $channel['name'] = trim($matches[1]);
        }
        
        // Extrair tvg-id
        if (preg_match('/tvg-id="([^"]+)"/', $line, $matches)) {
            $channel['id'] = $matches[1];
        }
        
        // Extrair tvg-logo
        if (preg_match('/tvg-logo="([^"]+)"/', $line, $matches)) {
            $channel['logo'] = $matches[1];
        }
        
        // Extrair group-title
        if (preg_match('/group-title="([^"]+)"/', $line, $matches)) {
            $channel['group'] = $matches[1];
        }
        
        return $channel;
    }
    
    private static function fetchContent($url) {
        $context = stream_context_create([
            'http' => [
                'timeout' => 30,
                'user_agent' => 'Neuro Play IPTV/1.0'
            ]
        ]);
        
        $content = @file_get_contents($url, false, $context);
        
        if ($content === false) {
            throw new Exception('Não foi possível acessar a URL');
        }
        
        return $content;
    }
    
    public static function filterByCategory($channels, $categoryId) {
        if (empty($categoryId)) {
            return $channels;
        }
        
        return array_filter($channels, function($channel) use ($categoryId) {
            return $channel['group'] === $categoryId || 
                   strpos($channel['group'], $categoryId) !== false;
        });
    }
}
