<?php
/**
 * Classe para gerenciamento de JWT (JSON Web Tokens)
 * Implementação simples sem biblioteca externa
 */
class JWT {
    
    public static function encode($payload, $expiration = null) {
        $header = [
            'typ' => 'JWT',
            'alg' => JWT_ALGORITHM
        ];
        
        if ($expiration === null) {
            $expiration = JWT_EXPIRATION;
        }
        
        $payload['iat'] = time();
        $payload['exp'] = time() + $expiration;
        
        $headerEncoded = self::base64UrlEncode(json_encode($header));
        $payloadEncoded = self::base64UrlEncode(json_encode($payload));
        
        $signature = hash_hmac('sha256', $headerEncoded . '.' . $payloadEncoded, JWT_SECRET_KEY, true);
        $signatureEncoded = self::base64UrlEncode($signature);
        
        return $headerEncoded . '.' . $payloadEncoded . '.' . $signatureEncoded;
    }
    
    public static function decode($token) {
        $parts = explode('.', $token);
        
        if (count($parts) !== 3) {
            throw new Exception('Token inválido');
        }
        
        list($headerEncoded, $payloadEncoded, $signatureEncoded) = $parts;
        
        $signature = self::base64UrlDecode($signatureEncoded);
        $expectedSignature = hash_hmac('sha256', $headerEncoded . '.' . $payloadEncoded, JWT_SECRET_KEY, true);
        
        if (!hash_equals($signature, $expectedSignature)) {
            throw new Exception('Assinatura inválida');
        }
        
        $payload = json_decode(self::base64UrlDecode($payloadEncoded), true);
        
        if (!isset($payload['exp']) || $payload['exp'] < time()) {
            throw new Exception('Token expirado');
        }
        
        return $payload;
    }
    
    public static function createRefreshToken($tipoUsuario, $usuarioId) {
        $token = bin2hex(random_bytes(32));
        $expiraEm = date('Y-m-d H:i:s', time() + JWT_REFRESH_EXPIRATION);
        
        try {
            $db = Database::getAdminConnection();
            
            $sql = "INSERT INTO refresh_tokens (tipo_usuario, usuario_id, token, expira_em, ip_address, user_agent) 
                    VALUES (:tipo_usuario, :usuario_id, :token, :expira_em, :ip_address, :user_agent)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'tipo_usuario' => $tipoUsuario,
                'usuario_id' => $usuarioId,
                'token' => $token,
                'expira_em' => $expiraEm,
                'ip_address' => $_SERVER['REMOTE_ADDR'] ?? null,
                'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? null
            ]);
            
            return $token;
        } catch (Exception $e) {
            throw new Exception('Erro ao criar refresh token');
        }
    }
    
    public static function validateRefreshToken($token) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM refresh_tokens 
                    WHERE token = :token 
                    AND revogado = 0 
                    AND expira_em > NOW()";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['token' => $token]);
            
            return $stmt->fetch();
        } catch (Exception $e) {
            return false;
        }
    }
    
    public static function revokeRefreshToken($token) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "UPDATE refresh_tokens SET revogado = 1 WHERE token = :token";
            $stmt = $db->prepare($sql);
            $stmt->execute(['token' => $token]);
            
            return true;
        } catch (Exception $e) {
            return false;
        }
    }
    
    private static function base64UrlEncode($data) {
        return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
    }
    
    private static function base64UrlDecode($data) {
        return base64_decode(strtr($data, '-_', '+/'));
    }
}
