<?php
/**
 * Classe para gerenciamento de upload de arquivos
 */
class FileUpload {
    
    public static function upload($file, $directory = 'general', $allowedTypes = null) {
        if (!isset($file['tmp_name']) || empty($file['tmp_name'])) {
            throw new Exception('Nenhum arquivo foi enviado');
        }
        
        if ($file['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('Erro no upload do arquivo');
        }
        
        if ($file['size'] > MAX_UPLOAD_SIZE) {
            throw new Exception('Arquivo muito grande. Tamanho máximo: ' . (MAX_UPLOAD_SIZE / 1024 / 1024) . 'MB');
        }
        
        $allowedTypes = $allowedTypes ?? ALLOWED_IMAGE_TYPES;
        
        $finfo = finfo_open(FILEINFO_MIME_TYPE);
        $mimeType = finfo_file($finfo, $file['tmp_name']);
        finfo_close($finfo);
        
        if (!in_array($mimeType, $allowedTypes)) {
            throw new Exception('Tipo de arquivo não permitido');
        }
        
        $extension = self::getExtensionFromMime($mimeType);
        $filename = self::generateUniqueFilename($extension);
        
        $uploadDir = UPLOAD_PATH . $directory . '/';
        
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $destination = $uploadDir . $filename;
        
        if (!move_uploaded_file($file['tmp_name'], $destination)) {
            throw new Exception('Erro ao salvar arquivo');
        }
        
        return $directory . '/' . $filename;
    }
    
    public static function delete($filepath) {
        $fullPath = UPLOAD_PATH . $filepath;
        
        if (file_exists($fullPath)) {
            return unlink($fullPath);
        }
        
        return false;
    }
    
    private static function generateUniqueFilename($extension) {
        return uniqid() . '_' . time() . '.' . $extension;
    }
    
    private static function getExtensionFromMime($mimeType) {
        $mimeMap = [
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp'
        ];
        
        return $mimeMap[$mimeType] ?? 'bin';
    }
    
    public static function handleUpload($fieldName, $directory = 'general') {
        if (!isset($_FILES[$fieldName])) {
            throw new Exception('Campo de arquivo não encontrado');
        }
        
        return self::upload($_FILES[$fieldName], $directory);
    }
}
