/**
 * Aplicação Principal
 */

const App = {
    currentPage: 'dashboard',
    
    init() {
        // Verificar autenticação
        if (!Auth.isAuthenticated()) {
            this.showLogin();
        } else {
            this.showDashboard();
        }
    },
    
    // Mostrar tela de login
    showLogin() {
        const app = document.getElementById('app');
        app.innerHTML = `
            <div class="login-container">
                <div class="login-box">
                    <h1>🎬 ${CONFIG.APP_NAME}</h1>
                    <p>Painel Administrativo</p>
                    
                    <div id="login-alert"></div>
                    
                    <form id="login-form">
                        <div class="form-group">
                            <label>Tipo de Usuário</label>
                            <select id="tipo" required>
                                <option value="admin">Administrador</option>
                                <option value="provedor">Provedor</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Email</label>
                            <input type="email" id="email" placeholder="seu@email.com" required>
                        </div>
                        
                        <div class="form-group">
                            <label>Senha</label>
                            <input type="password" id="senha" placeholder="••••••••" required>
                        </div>
                        
                        <button type="submit" class="btn btn-primary">
                            Entrar
                        </button>
                    </form>
                    
                    <p style="margin-top: 20px; text-align: center; font-size: 12px; color: #999;">
                        v${CONFIG.VERSION}
                    </p>
                </div>
            </div>
        `;
        
        // Event listener do formulário
        document.getElementById('login-form').addEventListener('submit', async (e) => {
            e.preventDefault();
            await this.handleLogin();
        });
    },
    
    // Processar login
    async handleLogin() {
        const email = document.getElementById('email').value;
        const senha = document.getElementById('senha').value;
        const tipo = document.getElementById('tipo').value;
        const alertDiv = document.getElementById('login-alert');
        
        alertDiv.innerHTML = '<div class="alert alert-success">Entrando...</div>';
        
        const result = await Auth.login(email, senha, tipo);
        
        if (result.success) {
            this.showDashboard();
        } else {
            alertDiv.innerHTML = `<div class="alert alert-error">${result.message}</div>`;
        }
    },
    
    // Mostrar dashboard
    showDashboard() {
        const user = Auth.getUser();
        const app = document.getElementById('app');
        
        app.innerHTML = `
            <div class="dashboard">
                <aside class="sidebar">
                    <div class="sidebar-header">
                        <h2>🎬 ${CONFIG.APP_NAME}</h2>
                        <p style="font-size: 12px; opacity: 0.7;">${user.tipo === 'admin' ? 'Administrador' : 'Provedor'}</p>
                    </div>
                    
                    <ul class="sidebar-menu" id="sidebar-menu">
                        ${user.tipo === 'admin' ? `
                            <li><a href="#" data-page="dashboard" class="active">📊 Dashboard</a></li>
                            <li><a href="#" data-page="provedores">🏢 Provedores</a></li>
                            <li><a href="#" data-page="configuracoes">⚙️ Configurações</a></li>
                            <li><a href="#" data-page="financeiro">💰 Financeiro</a></li>
                            <li><a href="#" data-page="logs">📋 Logs</a></li>
                        ` : `
                            <li><a href="#" data-page="dashboard" class="active">📊 Dashboard</a></li>
                            <li><a href="#" data-page="perfil">👤 Meu Perfil</a></li>
                            <li><a href="#" data-page="urls">🔗 URLs</a></li>
                            <li><a href="#" data-page="mensagens">💬 Mensagens</a></li>
                            <li><a href="#" data-page="novidades">🎉 Novidades</a></li>
                        `}
                    </ul>
                </aside>
                
                <main class="main-content">
                    <div class="header">
                        <h1 id="page-title">Dashboard</h1>
                        <div class="user-info">
                            <span>👤 ${user.nome}</span>
                            <button class="btn btn-danger btn-sm" onclick="Auth.logout()">Sair</button>
                        </div>
                    </div>
                    
                    <div id="page-content">
                        <!-- Conteúdo será carregado aqui -->
                    </div>
                </main>
            </div>
        `;
        
        // Event listeners do menu
        document.querySelectorAll('.sidebar-menu a').forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const page = e.target.dataset.page;
                this.loadPage(page);
                
                // Atualizar menu ativo
                document.querySelectorAll('.sidebar-menu a').forEach(l => l.classList.remove('active'));
                e.target.classList.add('active');
            });
        });
        
        // Carregar dashboard
        this.loadPage('dashboard');
    },
    
    // Carregar página
    async loadPage(page) {
        this.currentPage = page;
        const content = document.getElementById('page-content');
        const title = document.getElementById('page-title');
        
        content.innerHTML = '<div class="loading"><div class="spinner"></div><p>Carregando...</p></div>';
        
        switch(page) {
            case 'dashboard':
                title.textContent = 'Dashboard';
                await this.loadDashboard();
                break;
            case 'provedores':
                title.textContent = 'Gerenciar Provedores';
                await this.loadProvedores();
                break;
            case 'configuracoes':
                title.textContent = 'Configurações Gerais';
                await this.loadConfiguracoes();
                break;
            case 'financeiro':
                title.textContent = 'Controle Financeiro';
                await this.loadFinanceiro();
                break;
            case 'logs':
                title.textContent = 'Logs do Sistema';
                await this.loadLogs();
                break;
            default:
                content.innerHTML = '<div class="card"><p>Página em desenvolvimento...</p></div>';
        }
    },
    
    // Carregar dashboard
    async loadDashboard() {
        const content = document.getElementById('page-content');
        const user = Auth.getUser();
        
        if (user.tipo === 'admin') {
            // Dashboard Admin
            const provedores = await API.get('admin/provedores');
            const logs = await API.get('admin/logs?limit=5');
            
            content.innerHTML = `
                <div class="stats-grid">
                    <div class="stat-card">
                        <h3>Total de Provedores</h3>
                        <div class="value">${provedores.success ? provedores.data.length : 0}</div>
                    </div>
                    <div class="stat-card">
                        <h3>Provedores Ativos</h3>
                        <div class="value">${provedores.success ? provedores.data.filter(p => p.ativo).length : 0}</div>
                    </div>
                    <div class="stat-card">
                        <h3>Total de Logs</h3>
                        <div class="value">${logs.success ? logs.data.pagination.total : 0}</div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        <h2>Últimos Logs</h2>
                    </div>
                    <table>
                        <thead>
                            <tr>
                                <th>Data</th>
                                <th>Usuário</th>
                                <th>Ação</th>
                                <th>Descrição</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${logs.success && logs.data.logs.length > 0 ? 
                                logs.data.logs.map(log => `
                                    <tr>
                                        <td>${new Date(log.created_at).toLocaleString('pt-BR')}</td>
                                        <td>${log.tipo_usuario}</td>
                                        <td>${log.acao}</td>
                                        <td>${log.descricao}</td>
                                    </tr>
                                `).join('') : 
                                '<tr><td colspan="4" style="text-align: center;">Nenhum log encontrado</td></tr>'
                            }
                        </tbody>
                    </table>
                </div>
            `;
        } else {
            // Dashboard Provedor
            content.innerHTML = `
                <div class="card">
                    <h2>Bem-vindo, ${user.nome}!</h2>
                    <p>Painel do Provedor em desenvolvimento...</p>
                </div>
            `;
        }
    },
    
    // Carregar provedores
    async loadProvedores() {
        const content = document.getElementById('page-content');
        const result = await API.get('admin/provedores');
        
        if (!result.success) {
            content.innerHTML = `<div class="card"><div class="alert alert-error">${result.message}</div></div>`;
            return;
        }
        
        const provedores = result.data;
        
        content.innerHTML = `
            <div class="card">
                <div class="card-header">
                    <h2>Lista de Provedores</h2>
                    <button class="btn btn-primary" onclick="App.showModalProvedor()">+ Novo Provedor</button>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Número</th>
                            <th>Nome</th>
                            <th>Email</th>
                            <th>Status</th>
                            <th>Vencimento</th>
                            <th>Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${provedores.map(p => `
                            <tr>
                                <td>${p.id}</td>
                                <td>${p.numero_provedor}</td>
                                <td>${p.nome}</td>
                                <td>${p.email}</td>
                                <td>
                                    <span class="badge ${p.ativo ? 'badge-success' : 'badge-danger'}">
                                        ${p.ativo ? 'Ativo' : 'Inativo'}
                                    </span>
                                </td>
                                <td>${p.data_vencimento || '-'}</td>
                                <td class="actions">
                                    <button class="btn btn-secondary btn-sm" onclick="App.editProvedor(${p.id})">Editar</button>
                                    <button class="btn btn-danger btn-sm" onclick="App.deleteProvedor(${p.id})">Deletar</button>
                                </td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
    },
    
    // Carregar configurações
    async loadConfiguracoes() {
        const content = document.getElementById('page-content');
        const result = await API.get('admin/configuracoes');
        
        if (!result.success) {
            content.innerHTML = `<div class="card"><div class="alert alert-error">${result.message}</div></div>`;
            return;
        }
        
        content.innerHTML = `
            <div class="card">
                <div class="card-header">
                    <h2>Configurações do Sistema</h2>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>Chave</th>
                            <th>Valor</th>
                            <th>Descrição</th>
                            <th>Tipo</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${result.data.map(c => `
                            <tr>
                                <td><strong>${c.chave}</strong></td>
                                <td>${c.valor}</td>
                                <td>${c.descricao || '-'}</td>
                                <td>${c.tipo}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
    },
    
    // Carregar financeiro
    async loadFinanceiro() {
        const content = document.getElementById('page-content');
        const result = await API.get('admin/financeiro');
        
        if (!result.success) {
            content.innerHTML = `<div class="card"><div class="alert alert-error">${result.message}</div></div>`;
            return;
        }
        
        const { pagamentos, estatisticas } = result.data;
        
        content.innerHTML = `
            <div class="stats-grid">
                <div class="stat-card">
                    <h3>Total Pago</h3>
                    <div class="value" style="color: var(--success)">R$ ${parseFloat(estatisticas.total_pago || 0).toFixed(2)}</div>
                </div>
                <div class="stat-card">
                    <h3>Total Pendente</h3>
                    <div class="value" style="color: var(--warning)">R$ ${parseFloat(estatisticas.total_pendente || 0).toFixed(2)}</div>
                </div>
                <div class="stat-card">
                    <h3>Total Atrasado</h3>
                    <div class="value" style="color: var(--danger)">R$ ${parseFloat(estatisticas.total_atrasado || 0).toFixed(2)}</div>
                </div>
            </div>
            
            <div class="card">
                <div class="card-header">
                    <h2>Pagamentos</h2>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>Provedor</th>
                            <th>Valor</th>
                            <th>Vencimento</th>
                            <th>Status</th>
                            <th>Pagamento</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${pagamentos.map(p => `
                            <tr>
                                <td>${p.provedor_nome}</td>
                                <td>R$ ${parseFloat(p.valor).toFixed(2)}</td>
                                <td>${new Date(p.data_vencimento).toLocaleDateString('pt-BR')}</td>
                                <td>
                                    <span class="badge badge-${p.status === 'pago' ? 'success' : p.status === 'pendente' ? 'warning' : 'danger'}">
                                        ${p.status}
                                    </span>
                                </td>
                                <td>${p.data_pagamento ? new Date(p.data_pagamento).toLocaleDateString('pt-BR') : '-'}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
    },
    
    // Carregar logs
    async loadLogs() {
        const content = document.getElementById('page-content');
        const result = await API.get('admin/logs?limit=50');
        
        if (!result.success) {
            content.innerHTML = `<div class="card"><div class="alert alert-error">${result.message}</div></div>`;
            return;
        }
        
        content.innerHTML = `
            <div class="card">
                <div class="card-header">
                    <h2>Logs do Sistema</h2>
                </div>
                
                <table>
                    <thead>
                        <tr>
                            <th>Data/Hora</th>
                            <th>Tipo</th>
                            <th>Usuário ID</th>
                            <th>Ação</th>
                            <th>Descrição</th>
                            <th>IP</th>
                        </tr>
                    </thead>
                    <tbody>
                        ${result.data.logs.map(log => `
                            <tr>
                                <td>${new Date(log.created_at).toLocaleString('pt-BR')}</td>
                                <td>${log.tipo_usuario}</td>
                                <td>${log.usuario_id}</td>
                                <td>${log.acao}</td>
                                <td>${log.descricao}</td>
                                <td>${log.ip_address || '-'}</td>
                            </tr>
                        `).join('')}
                    </tbody>
                </table>
            </div>
        `;
    }
};

// Inicializar aplicação quando DOM carregar
document.addEventListener('DOMContentLoaded', () => {
    App.init();
});
