-- =============================================
-- Database: neuroplay_users
-- Descrição: Banco de dados para Usuários e Perfis
-- =============================================

CREATE DATABASE IF NOT EXISTS soupet_provedor CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE soupet_provedor;

-- =============================================
-- Tabela: usuarios
-- Descrição: Usuários do sistema (autenticados via provedor)
-- =============================================
CREATE TABLE IF NOT EXISTS usuarios (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    provedor_id INT UNSIGNED NOT NULL COMMENT 'ID do provedor (referência externa)',
    numero_provedor VARCHAR(20) NOT NULL COMMENT 'Número do provedor',
    login_provedor VARCHAR(150) NOT NULL COMMENT 'Login no provedor IPTV',
    nome VARCHAR(150) NULL,
    email VARCHAR(150) NULL,
    idioma VARCHAR(5) DEFAULT 'pt-BR',
    controle_parental_ativo TINYINT(1) DEFAULT 0,
    pin_parental VARCHAR(6) NULL COMMENT 'PIN de 4-6 dígitos',
    ativo TINYINT(1) DEFAULT 1,
    ultimo_acesso DATETIME NULL,
    dados_provedor JSON NULL COMMENT 'Dados retornados pelo provedor',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_usuario_provedor (provedor_id, login_provedor),
    INDEX idx_provedor_id (provedor_id),
    INDEX idx_numero_provedor (numero_provedor),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: perfis
-- Descrição: Perfis dos usuários (1 principal + até 4 adicionais)
-- =============================================
CREATE TABLE IF NOT EXISTS perfis (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    usuario_id BIGINT UNSIGNED NOT NULL,
    nome VARCHAR(100) NOT NULL,
    avatar VARCHAR(255) NULL COMMENT 'ID ou caminho do avatar',
    tipo ENUM('principal', 'adicional') DEFAULT 'adicional',
    pin_protegido TINYINT(1) DEFAULT 0,
    pin VARCHAR(6) NULL COMMENT 'PIN específico do perfil',
    infantil TINYINT(1) DEFAULT 0 COMMENT 'Perfil infantil (só acessa conteúdo infantil)',
    ativo TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    INDEX idx_usuario_id (usuario_id),
    INDEX idx_tipo (tipo),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: favoritos
-- Descrição: Conteúdos favoritos dos perfis
-- =============================================
CREATE TABLE IF NOT EXISTS favoritos (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    perfil_id BIGINT UNSIGNED NOT NULL,
    tipo_conteudo ENUM('canal', 'filme', 'serie') NOT NULL,
    conteudo_id VARCHAR(100) NOT NULL COMMENT 'ID do conteúdo no provedor',
    nome VARCHAR(255) NOT NULL,
    imagem VARCHAR(500) NULL,
    dados_adicionais JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (perfil_id) REFERENCES perfis(id) ON DELETE CASCADE,
    UNIQUE KEY unique_favorito (perfil_id, tipo_conteudo, conteudo_id),
    INDEX idx_perfil_id (perfil_id),
    INDEX idx_tipo_conteudo (tipo_conteudo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: historico_visualizacao
-- Descrição: Histórico de visualização (continuar assistindo)
-- =============================================
CREATE TABLE IF NOT EXISTS historico_visualizacao (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    perfil_id BIGINT UNSIGNED NOT NULL,
    tipo_conteudo ENUM('canal', 'filme', 'serie') NOT NULL,
    conteudo_id VARCHAR(100) NOT NULL COMMENT 'ID do conteúdo no provedor',
    nome VARCHAR(255) NOT NULL,
    imagem VARCHAR(500) NULL,
    posicao_segundos INT UNSIGNED DEFAULT 0 COMMENT 'Posição em segundos',
    duracao_segundos INT UNSIGNED DEFAULT 0 COMMENT 'Duração total em segundos',
    percentual_assistido DECIMAL(5, 2) DEFAULT 0.00,
    episodio_id VARCHAR(100) NULL COMMENT 'ID do episódio (para séries)',
    temporada INT NULL COMMENT 'Número da temporada (para séries)',
    episodio INT NULL COMMENT 'Número do episódio (para séries)',
    finalizado TINYINT(1) DEFAULT 0,
    dados_adicionais JSON NULL,
    ultima_visualizacao DATETIME DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (perfil_id) REFERENCES perfis(id) ON DELETE CASCADE,
    UNIQUE KEY unique_historico (perfil_id, tipo_conteudo, conteudo_id, episodio_id),
    INDEX idx_perfil_id (perfil_id),
    INDEX idx_tipo_conteudo (tipo_conteudo),
    INDEX idx_ultima_visualizacao (ultima_visualizacao),
    INDEX idx_finalizado (finalizado)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: recomendacoes
-- Descrição: Cache de recomendações por perfil
-- =============================================
CREATE TABLE IF NOT EXISTS recomendacoes (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    perfil_id BIGINT UNSIGNED NOT NULL,
    tipo_conteudo ENUM('canal', 'filme', 'serie') NOT NULL,
    conteudo_id VARCHAR(100) NOT NULL,
    nome VARCHAR(255) NOT NULL,
    imagem VARCHAR(500) NULL,
    score DECIMAL(5, 2) DEFAULT 0.00 COMMENT 'Score de recomendação',
    motivo VARCHAR(100) NULL COMMENT 'Motivo da recomendação',
    dados_adicionais JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (perfil_id) REFERENCES perfis(id) ON DELETE CASCADE,
    INDEX idx_perfil_id (perfil_id),
    INDEX idx_score (score DESC)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: avatares
-- Descrição: Avatares disponíveis no sistema
-- =============================================
CREATE TABLE IF NOT EXISTS avatares (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    arquivo VARCHAR(255) NOT NULL COMMENT 'Caminho do arquivo',
    categoria ENUM('geral', 'infantil', 'animais', 'personagens') DEFAULT 'geral',
    ativo TINYINT(1) DEFAULT 1,
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_categoria (categoria),
    INDEX idx_ativo (ativo),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: cache_playlist
-- Descrição: Cache de playlists M3U processadas
-- =============================================
CREATE TABLE IF NOT EXISTS cache_playlist (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    provedor_id INT UNSIGNED NOT NULL,
    tipo ENUM('canais', 'filmes', 'series') NOT NULL,
    dados JSON NOT NULL COMMENT 'Dados processados da playlist',
    hash VARCHAR(64) NOT NULL COMMENT 'Hash MD5 da playlist original',
    expira_em DATETIME NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    UNIQUE KEY unique_cache (provedor_id, tipo),
    INDEX idx_provedor_id (provedor_id),
    INDEX idx_expira_em (expira_em)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Inserir Avatares Padrão
-- =============================================
INSERT INTO avatares (nome, arquivo, categoria, ordem) VALUES
('Avatar 1', 'avatar_01.png', 'geral', 1),
('Avatar 2', 'avatar_02.png', 'geral', 2),
('Avatar 3', 'avatar_03.png', 'geral', 3),
('Avatar 4', 'avatar_04.png', 'geral', 4),
('Avatar 5', 'avatar_05.png', 'geral', 5),
('Criança 1', 'avatar_kid_01.png', 'infantil', 6),
('Criança 2', 'avatar_kid_02.png', 'infantil', 7),
('Criança 3', 'avatar_kid_03.png', 'infantil', 8);
