-- =============================================
-- Database: neuroplay_admin
-- Descrição: Banco de dados para Admin e Provedores
-- =============================================

CREATE DATABASE IF NOT EXISTS soupet_gerencia CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci;
USE soupet_gerencia;

-- =============================================
-- Tabela: admins
-- Descrição: Administradores do sistema
-- =============================================
CREATE TABLE IF NOT EXISTS admins (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(150) NOT NULL UNIQUE,
    senha VARCHAR(255) NOT NULL,
    ativo TINYINT(1) DEFAULT 1,
    ultimo_acesso DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: provedores
-- Descrição: Provedores IPTV cadastrados
-- =============================================
CREATE TABLE IF NOT EXISTS provedores (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    numero_provedor VARCHAR(20) NOT NULL UNIQUE COMMENT 'Número único do provedor para login',
    nome VARCHAR(150) NOT NULL,
    email VARCHAR(150) NOT NULL UNIQUE,
    senha VARCHAR(255) NOT NULL,
    logo VARCHAR(255) NULL COMMENT 'Caminho do arquivo de logo',
    banner VARCHAR(255) NULL COMMENT 'Caminho do arquivo de banner',
    url_principal VARCHAR(500) NOT NULL COMMENT 'URL principal do provedor',
    url_backup_1 VARCHAR(500) NULL COMMENT 'URL de backup 1',
    url_backup_2 VARCHAR(500) NULL COMMENT 'URL de backup 2',
    categoria_adulto_id VARCHAR(50) NULL COMMENT 'ID da categoria adulto no provedor',
    categoria_infantil_id VARCHAR(50) NULL COMMENT 'ID da categoria infantil no provedor',
    ativo TINYINT(1) DEFAULT 1,
    data_vencimento DATE NULL COMMENT 'Data de vencimento do pagamento',
    ultimo_acesso DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_numero_provedor (numero_provedor),
    INDEX idx_email (email),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: configuracoes_gerais
-- Descrição: Configurações globais do sistema
-- =============================================
CREATE TABLE IF NOT EXISTS configuracoes_gerais (
    id INT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    chave VARCHAR(100) NOT NULL UNIQUE,
    valor TEXT NULL,
    descricao VARCHAR(255) NULL,
    tipo ENUM('string', 'number', 'boolean', 'json') DEFAULT 'string',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_chave (chave)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: logs_sistema
-- Descrição: Logs de ações do sistema
-- =============================================
CREATE TABLE IF NOT EXISTS logs_sistema (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    tipo_usuario ENUM('admin', 'provedor') NOT NULL,
    usuario_id INT UNSIGNED NOT NULL,
    acao VARCHAR(100) NOT NULL COMMENT 'Tipo de ação realizada',
    descricao TEXT NULL,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    dados_adicionais JSON NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_tipo_usuario (tipo_usuario, usuario_id),
    INDEX idx_acao (acao),
    INDEX idx_created_at (created_at)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: controle_financeiro
-- Descrição: Controle de pagamentos dos provedores
-- =============================================
CREATE TABLE IF NOT EXISTS controle_financeiro (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    provedor_id INT UNSIGNED NOT NULL,
    valor DECIMAL(10, 2) NOT NULL,
    data_vencimento DATE NOT NULL,
    data_pagamento DATE NULL,
    status ENUM('pendente', 'pago', 'atrasado', 'cancelado') DEFAULT 'pendente',
    metodo_pagamento VARCHAR(50) NULL,
    comprovante VARCHAR(255) NULL COMMENT 'Caminho do comprovante',
    observacoes TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (provedor_id) REFERENCES provedores(id) ON DELETE CASCADE,
    INDEX idx_provedor_id (provedor_id),
    INDEX idx_status (status),
    INDEX idx_data_vencimento (data_vencimento)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: mensagens_provedor
-- Descrição: Mensagens cadastradas pelos provedores
-- =============================================
CREATE TABLE IF NOT EXISTS mensagens_provedor (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    provedor_id INT UNSIGNED NOT NULL,
    titulo VARCHAR(200) NOT NULL,
    conteudo TEXT NOT NULL,
    tipo ENUM('info', 'aviso', 'urgente') DEFAULT 'info',
    ativo TINYINT(1) DEFAULT 1,
    data_inicio DATETIME NULL,
    data_fim DATETIME NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (provedor_id) REFERENCES provedores(id) ON DELETE CASCADE,
    INDEX idx_provedor_id (provedor_id),
    INDEX idx_ativo (ativo),
    INDEX idx_datas (data_inicio, data_fim)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: novidades_provedor
-- Descrição: Novidades cadastradas pelos provedores
-- =============================================
CREATE TABLE IF NOT EXISTS novidades_provedor (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    provedor_id INT UNSIGNED NOT NULL,
    titulo VARCHAR(200) NOT NULL,
    descricao TEXT NULL,
    imagem VARCHAR(255) NULL,
    link VARCHAR(500) NULL,
    destaque TINYINT(1) DEFAULT 0,
    ativo TINYINT(1) DEFAULT 1,
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (provedor_id) REFERENCES provedores(id) ON DELETE CASCADE,
    INDEX idx_provedor_id (provedor_id),
    INDEX idx_ativo (ativo),
    INDEX idx_destaque (destaque),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Tabela: refresh_tokens
-- Descrição: Tokens de refresh para JWT
-- =============================================
CREATE TABLE IF NOT EXISTS refresh_tokens (
    id BIGINT UNSIGNED AUTO_INCREMENT PRIMARY KEY,
    tipo_usuario ENUM('admin', 'provedor') NOT NULL,
    usuario_id INT UNSIGNED NOT NULL,
    token VARCHAR(500) NOT NULL UNIQUE,
    expira_em DATETIME NOT NULL,
    revogado TINYINT(1) DEFAULT 0,
    ip_address VARCHAR(45) NULL,
    user_agent TEXT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_token (token),
    INDEX idx_usuario (tipo_usuario, usuario_id),
    INDEX idx_expira_em (expira_em)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- Inserir Admin Padrão
-- Senha: admin123 (ALTERAR EM PRODUÇÃO)
-- =============================================
INSERT INTO admins (nome, email, senha) VALUES 
('Administrador', 'admin@neuroplay.com', '$2y$10$92IXUNpkjO0rOQ5byMi.Ye4oKoEa3Ro9llC/.og/at2.uheWG/igi');

-- =============================================
-- Inserir Configurações Padrão
-- =============================================
INSERT INTO configuracoes_gerais (chave, valor, descricao, tipo) VALUES
('categoria_adulto_ativa', '0', 'Ativar categoria adulto no sistema', 'boolean'),
('manutencao', '0', 'Modo de manutenção', 'boolean'),
('versao_app', '1.0.0', 'Versão do aplicativo', 'string'),
('max_perfis_por_usuario', '4', 'Máximo de perfis por usuário', 'number');
