<?php
/**
 * Controller de Usuários (Frontend)
 */
class UsuarioController {
    
    public function route($url, $method) {
        $action = $url[1] ?? 'index';
        
        switch ($action) {
            case 'login':
                if ($method === 'POST') {
                    $this->login();
                }
                break;
                
            case 'provedores':
                if ($method === 'GET') {
                    $this->listProvedores();
                }
                break;
                
            case 'configuracoes':
                $this->handleConfiguracoes($method);
                break;
                
            default:
                Response::notFound();
        }
    }
    
    private function login() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'numero_provedor' => 'required',
            'login' => 'required',
            'senha' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            // Buscar provedor
            $dbAdmin = Database::getAdminConnection();
            
            $sqlProvedor = "SELECT * FROM provedores WHERE numero_provedor = :numero_provedor AND ativo = 1";
            $stmtProvedor = $dbAdmin->prepare($sqlProvedor);
            $stmtProvedor->execute(['numero_provedor' => $data['numero_provedor']]);
            
            $provedor = $stmtProvedor->fetch();
            
            if (!$provedor) {
                Response::error('Provedor não encontrado ou inativo', 404);
            }
            
            // Autenticar no provedor IPTV (simulado - implementar integração real)
            $authResult = $this->authenticateWithProvider(
                $provedor['url_principal'],
                $data['login'],
                $data['senha']
            );
            
            if (!$authResult['success']) {
                Response::error('Credenciais inválidas no provedor', 401);
            }
            
            // Buscar ou criar usuário
            $dbUsers = Database::getUsersConnection();
            
            $sqlUser = "SELECT * FROM usuarios 
                        WHERE provedor_id = :provedor_id AND login_provedor = :login_provedor";
            
            $stmtUser = $dbUsers->prepare($sqlUser);
            $stmtUser->execute([
                'provedor_id' => $provedor['id'],
                'login_provedor' => $data['login']
            ]);
            
            $usuario = $stmtUser->fetch();
            
            if (!$usuario) {
                // Criar novo usuário
                $sqlInsert = "INSERT INTO usuarios (provedor_id, numero_provedor, login_provedor, nome, dados_provedor) 
                              VALUES (:provedor_id, :numero_provedor, :login_provedor, :nome, :dados_provedor)";
                
                $stmtInsert = $dbUsers->prepare($sqlInsert);
                $stmtInsert->execute([
                    'provedor_id' => $provedor['id'],
                    'numero_provedor' => $data['numero_provedor'],
                    'login_provedor' => $data['login'],
                    'nome' => $authResult['user_data']['nome'] ?? $data['login'],
                    'dados_provedor' => json_encode($authResult['user_data'])
                ]);
                
                $usuarioId = $dbUsers->lastInsertId();
                
                // Criar perfil principal
                $sqlPerfil = "INSERT INTO perfis (usuario_id, nome, tipo) VALUES (:usuario_id, :nome, 'principal')";
                $stmtPerfil = $dbUsers->prepare($sqlPerfil);
                $stmtPerfil->execute([
                    'usuario_id' => $usuarioId,
                    'nome' => 'Principal'
                ]);
                
                $usuario = [
                    'id' => $usuarioId,
                    'provedor_id' => $provedor['id'],
                    'numero_provedor' => $data['numero_provedor'],
                    'login_provedor' => $data['login'],
                    'nome' => $authResult['user_data']['nome'] ?? $data['login']
                ];
            } else {
                // Atualizar último acesso
                $sqlUpdate = "UPDATE usuarios SET ultimo_acesso = NOW() WHERE id = :id";
                $stmtUpdate = $dbUsers->prepare($sqlUpdate);
                $stmtUpdate->execute(['id' => $usuario['id']]);
            }
            
            // Buscar perfis do usuário
            $sqlPerfis = "SELECT id, nome, avatar, tipo, pin_protegido, infantil FROM perfis 
                          WHERE usuario_id = :usuario_id AND ativo = 1";
            
            $stmtPerfis = $dbUsers->prepare($sqlPerfis);
            $stmtPerfis->execute(['usuario_id' => $usuario['id']]);
            $perfis = $stmtPerfis->fetchAll();
            
            Response::success([
                'usuario' => [
                    'id' => $usuario['id'],
                    'nome' => $usuario['nome'],
                    'provedor_id' => $usuario['provedor_id'],
                    'numero_provedor' => $usuario['numero_provedor']
                ],
                'provedor' => [
                    'id' => $provedor['id'],
                    'nome' => $provedor['nome'],
                    'logo' => $provedor['logo'],
                    'banner' => $provedor['banner']
                ],
                'perfis' => $perfis
            ], 'Login realizado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao realizar login: ' . $e->getMessage());
        }
    }
    
    private function authenticateWithProvider($url, $login, $senha) {
        // Implementar integração real com provedor IPTV
        // Por enquanto, retorna sucesso simulado
        
        // Exemplo de integração:
        // 1. Fazer requisição para API do provedor
        // 2. Validar credenciais
        // 3. Retornar dados do usuário
        
        return [
            'success' => true,
            'user_data' => [
                'nome' => $login,
                'status' => 'ativo',
                'expiracao' => date('Y-m-d', strtotime('+30 days'))
            ]
        ];
    }
    
    private function listProvedores() {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT id, numero_provedor, nome, logo FROM provedores WHERE ativo = 1 ORDER BY nome";
            $stmt = $db->query($sql);
            
            $provedores = $stmt->fetchAll();
            
            Response::success($provedores);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar provedores: ' . $e->getMessage());
        }
    }
    
    private function handleConfiguracoes($method) {
        // Implementar configurações do usuário
        Response::success(['message' => 'Configurações de usuário - Em desenvolvimento']);
    }
}
