<?php
/**
 * Controller do Painel Provedores
 */
class ProvedorController {
    
    public function route($url, $method) {
        // Autenticar como provedor
        $auth = AuthMiddleware::authenticateProvedor();
        
        $action = $url[1] ?? 'index';
        
        switch ($action) {
            case 'perfil':
                $this->handlePerfil($method, $auth);
                break;
                
            case 'urls':
                $this->handleUrls($method, $auth);
                break;
                
            case 'configuracoes':
                $this->handleConfiguracoes($method, $auth);
                break;
                
            case 'categorias':
                $this->handleCategorias($method, $auth);
                break;
                
            case 'mensagens':
                $this->handleMensagens($url, $method, $auth);
                break;
                
            case 'novidades':
                $this->handleNovidades($url, $method, $auth);
                break;
                
            case 'logs':
                $this->handleLogs($method, $auth);
                break;
                
            default:
                Response::notFound();
        }
    }
    
    private function handlePerfil($method, $auth) {
        if ($method === 'GET') {
            try {
                $db = Database::getAdminConnection();
                
                $sql = "SELECT id, numero_provedor, nome, email, logo, banner, url_principal, 
                        url_backup_1, url_backup_2, categoria_adulto_id, categoria_infantil_id, 
                        ativo, data_vencimento, created_at 
                        FROM provedores WHERE id = :id";
                
                $stmt = $db->prepare($sql);
                $stmt->execute(['id' => $auth['id']]);
                
                $provedor = $stmt->fetch();
                
                Response::success($provedor);
                
            } catch (Exception $e) {
                Response::error('Erro ao buscar perfil: ' . $e->getMessage());
            }
        }
    }
    
    private function handleUrls($method, $auth) {
        if ($method === 'PUT') {
            $data = json_decode(file_get_contents('php://input'), true);
            
            $validator = new Validator();
            if (!$validator->validate($data, [
                'url_principal' => 'required'
            ])) {
                Response::error('Dados inválidos', 400, $validator->getErrors());
            }
            
            try {
                $db = Database::getAdminConnection();
                
                $sql = "UPDATE provedores 
                        SET url_principal = :url_principal, 
                            url_backup_1 = :url_backup_1, 
                            url_backup_2 = :url_backup_2 
                        WHERE id = :id";
                
                $stmt = $db->prepare($sql);
                $stmt->execute([
                    'url_principal' => $data['url_principal'],
                    'url_backup_1' => $data['url_backup_1'] ?? null,
                    'url_backup_2' => $data['url_backup_2'] ?? null,
                    'id' => $auth['id']
                ]);
                
                Logger::log('provedor', $auth['id'], 'atualizar_urls', 'URLs atualizadas');
                
                Response::success(null, 'URLs atualizadas com sucesso');
                
            } catch (Exception $e) {
                Response::error('Erro ao atualizar URLs: ' . $e->getMessage());
            }
        }
    }
    
    private function handleConfiguracoes($method, $auth) {
        if ($method === 'PUT') {
            $data = json_decode(file_get_contents('php://input'), true);
            
            try {
                $db = Database::getAdminConnection();
                
                $fields = [];
                $params = ['id' => $auth['id']];
                
                if (isset($data['nome'])) {
                    $fields[] = "nome = :nome";
                    $params['nome'] = $data['nome'];
                }
                
                if (isset($data['logo'])) {
                    $fields[] = "logo = :logo";
                    $params['logo'] = $data['logo'];
                }
                
                if (isset($data['banner'])) {
                    $fields[] = "banner = :banner";
                    $params['banner'] = $data['banner'];
                }
                
                if (empty($fields)) {
                    Response::error('Nenhum campo para atualizar', 400);
                }
                
                $sql = "UPDATE provedores SET " . implode(', ', $fields) . " WHERE id = :id";
                $stmt = $db->prepare($sql);
                $stmt->execute($params);
                
                Logger::log('provedor', $auth['id'], 'atualizar_configuracoes', 'Configurações atualizadas');
                
                Response::success(null, 'Configurações atualizadas com sucesso');
                
            } catch (Exception $e) {
                Response::error('Erro ao atualizar configurações: ' . $e->getMessage());
            }
        }
    }
    
    private function handleCategorias($method, $auth) {
        if ($method === 'PUT') {
            $data = json_decode(file_get_contents('php://input'), true);
            
            try {
                $db = Database::getAdminConnection();
                
                $sql = "UPDATE provedores 
                        SET categoria_adulto_id = :categoria_adulto_id, 
                            categoria_infantil_id = :categoria_infantil_id 
                        WHERE id = :id";
                
                $stmt = $db->prepare($sql);
                $stmt->execute([
                    'categoria_adulto_id' => $data['categoria_adulto_id'] ?? null,
                    'categoria_infantil_id' => $data['categoria_infantil_id'] ?? null,
                    'id' => $auth['id']
                ]);
                
                Logger::log('provedor', $auth['id'], 'atualizar_categorias', 'Categorias atualizadas');
                
                Response::success(null, 'Categorias atualizadas com sucesso');
                
            } catch (Exception $e) {
                Response::error('Erro ao atualizar categorias: ' . $e->getMessage());
            }
        }
    }
    
    private function handleMensagens($url, $method, $auth) {
        $id = $url[2] ?? null;
        
        switch ($method) {
            case 'GET':
                $this->listMensagens($auth);
                break;
                
            case 'POST':
                $this->createMensagem($auth);
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateMensagem($id, $auth);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteMensagem($id, $auth);
                }
                break;
        }
    }
    
    private function listMensagens($auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM mensagens_provedor 
                    WHERE provedor_id = :provedor_id 
                    ORDER BY created_at DESC";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['provedor_id' => $auth['id']]);
            
            $mensagens = $stmt->fetchAll();
            
            Response::success($mensagens);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar mensagens: ' . $e->getMessage());
        }
    }
    
    private function createMensagem($auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'titulo' => 'required',
            'conteudo' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getAdminConnection();
            
            $sql = "INSERT INTO mensagens_provedor (provedor_id, titulo, conteudo, tipo, data_inicio, data_fim) 
                    VALUES (:provedor_id, :titulo, :conteudo, :tipo, :data_inicio, :data_fim)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'provedor_id' => $auth['id'],
                'titulo' => $data['titulo'],
                'conteudo' => $data['conteudo'],
                'tipo' => $data['tipo'] ?? 'info',
                'data_inicio' => $data['data_inicio'] ?? null,
                'data_fim' => $data['data_fim'] ?? null
            ]);
            
            $mensagemId = $db->lastInsertId();
            
            Logger::log('provedor', $auth['id'], 'criar_mensagem', "Mensagem criada: {$data['titulo']}");
            
            Response::success(['id' => $mensagemId], 'Mensagem criada com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar mensagem: ' . $e->getMessage());
        }
    }
    
    private function updateMensagem($id, $auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            $db = Database::getAdminConnection();
            
            $fields = [];
            $params = ['id' => $id, 'provedor_id' => $auth['id']];
            
            $allowedFields = ['titulo', 'conteudo', 'tipo', 'ativo', 'data_inicio', 'data_fim'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            $sql = "UPDATE mensagens_provedor SET " . implode(', ', $fields) . " 
                    WHERE id = :id AND provedor_id = :provedor_id";
            
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Logger::log('provedor', $auth['id'], 'atualizar_mensagem', "Mensagem ID {$id} atualizada");
            
            Response::success(null, 'Mensagem atualizada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar mensagem: ' . $e->getMessage());
        }
    }
    
    private function deleteMensagem($id, $auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "DELETE FROM mensagens_provedor WHERE id = :id AND provedor_id = :provedor_id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id, 'provedor_id' => $auth['id']]);
            
            Logger::log('provedor', $auth['id'], 'deletar_mensagem', "Mensagem ID {$id} deletada");
            
            Response::success(null, 'Mensagem deletada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar mensagem: ' . $e->getMessage());
        }
    }
    
    private function handleNovidades($url, $method, $auth) {
        $id = $url[2] ?? null;
        
        switch ($method) {
            case 'GET':
                $this->listNovidades($auth);
                break;
                
            case 'POST':
                $this->createNovidade($auth);
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateNovidade($id, $auth);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteNovidade($id, $auth);
                }
                break;
        }
    }
    
    private function listNovidades($auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM novidades_provedor 
                    WHERE provedor_id = :provedor_id 
                    ORDER BY destaque DESC, ordem ASC, created_at DESC";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['provedor_id' => $auth['id']]);
            
            $novidades = $stmt->fetchAll();
            
            Response::success($novidades);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar novidades: ' . $e->getMessage());
        }
    }
    
    private function createNovidade($auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'titulo' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getAdminConnection();
            
            $sql = "INSERT INTO novidades_provedor (provedor_id, titulo, descricao, imagem, link, destaque, ordem) 
                    VALUES (:provedor_id, :titulo, :descricao, :imagem, :link, :destaque, :ordem)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'provedor_id' => $auth['id'],
                'titulo' => $data['titulo'],
                'descricao' => $data['descricao'] ?? null,
                'imagem' => $data['imagem'] ?? null,
                'link' => $data['link'] ?? null,
                'destaque' => $data['destaque'] ?? 0,
                'ordem' => $data['ordem'] ?? 0
            ]);
            
            $novidadeId = $db->lastInsertId();
            
            Logger::log('provedor', $auth['id'], 'criar_novidade', "Novidade criada: {$data['titulo']}");
            
            Response::success(['id' => $novidadeId], 'Novidade criada com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar novidade: ' . $e->getMessage());
        }
    }
    
    private function updateNovidade($id, $auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            $db = Database::getAdminConnection();
            
            $fields = [];
            $params = ['id' => $id, 'provedor_id' => $auth['id']];
            
            $allowedFields = ['titulo', 'descricao', 'imagem', 'link', 'destaque', 'ativo', 'ordem'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            $sql = "UPDATE novidades_provedor SET " . implode(', ', $fields) . " 
                    WHERE id = :id AND provedor_id = :provedor_id";
            
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Logger::log('provedor', $auth['id'], 'atualizar_novidade', "Novidade ID {$id} atualizada");
            
            Response::success(null, 'Novidade atualizada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar novidade: ' . $e->getMessage());
        }
    }
    
    private function deleteNovidade($id, $auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "DELETE FROM novidades_provedor WHERE id = :id AND provedor_id = :provedor_id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id, 'provedor_id' => $auth['id']]);
            
            Logger::log('provedor', $auth['id'], 'deletar_novidade', "Novidade ID {$id} deletada");
            
            Response::success(null, 'Novidade deletada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar novidade: ' . $e->getMessage());
        }
    }
    
    private function handleLogs($method, $auth) {
        if ($method === 'GET') {
            try {
                $db = Database::getAdminConnection();
                
                $sql = "SELECT * FROM logs_sistema 
                        WHERE tipo_usuario = 'provedor' AND usuario_id = :usuario_id 
                        ORDER BY created_at DESC 
                        LIMIT 100";
                
                $stmt = $db->prepare($sql);
                $stmt->execute(['usuario_id' => $auth['id']]);
                
                $logs = $stmt->fetchAll();
                
                Response::success($logs);
                
            } catch (Exception $e) {
                Response::error('Erro ao buscar logs: ' . $e->getMessage());
            }
        }
    }
}
