<?php
/**
 * Controller de Perfis
 */
class PerfilController {
    
    public function route($url, $method) {
        $action = $url[1] ?? 'index';
        
        switch ($action) {
            case 'list':
                if ($method === 'GET') {
                    $this->listPerfis();
                }
                break;
                
            case 'create':
                if ($method === 'POST') {
                    $this->createPerfil();
                }
                break;
                
            case 'update':
                if ($method === 'PUT') {
                    $this->updatePerfil();
                }
                break;
                
            case 'delete':
                if ($method === 'DELETE') {
                    $this->deletePerfil();
                }
                break;
                
            case 'favoritos':
                $this->handleFavoritos($url, $method);
                break;
                
            case 'historico':
                $this->handleHistorico($url, $method);
                break;
                
            case 'recomendacoes':
                if ($method === 'GET') {
                    $this->getRecomendacoes();
                }
                break;
                
            case 'avatares':
                if ($method === 'GET') {
                    $this->listAvatares();
                }
                break;
                
            default:
                Response::notFound();
        }
    }
    
    private function listPerfis() {
        $usuarioId = $_GET['usuario_id'] ?? null;
        
        if (!$usuarioId) {
            Response::error('ID do usuário não fornecido', 400);
        }
        
        try {
            $db = Database::getUsersConnection();
            
            $sql = "SELECT id, nome, avatar, tipo, pin_protegido, infantil FROM perfis 
                    WHERE usuario_id = :usuario_id AND ativo = 1";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['usuario_id' => $usuarioId]);
            
            $perfis = $stmt->fetchAll();
            
            Response::success($perfis);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar perfis: ' . $e->getMessage());
        }
    }
    
    private function createPerfil() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'usuario_id' => 'required|numeric',
            'nome' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getUsersConnection();
            
            // Verificar limite de perfis
            $sqlCount = "SELECT COUNT(*) as total FROM perfis WHERE usuario_id = :usuario_id AND ativo = 1";
            $stmtCount = $db->prepare($sqlCount);
            $stmtCount->execute(['usuario_id' => $data['usuario_id']]);
            $count = $stmtCount->fetch()['total'];
            
            if ($count >= 5) { // 1 principal + 4 adicionais
                Response::error('Limite de perfis atingido', 400);
            }
            
            $sql = "INSERT INTO perfis (usuario_id, nome, avatar, pin_protegido, pin, infantil) 
                    VALUES (:usuario_id, :nome, :avatar, :pin_protegido, :pin, :infantil)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'usuario_id' => $data['usuario_id'],
                'nome' => $data['nome'],
                'avatar' => $data['avatar'] ?? null,
                'pin_protegido' => $data['pin_protegido'] ?? 0,
                'pin' => isset($data['pin']) ? password_hash($data['pin'], PASSWORD_BCRYPT) : null,
                'infantil' => $data['infantil'] ?? 0
            ]);
            
            $perfilId = $db->lastInsertId();
            
            Response::success(['id' => $perfilId], 'Perfil criado com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar perfil: ' . $e->getMessage());
        }
    }
    
    private function updatePerfil() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            Response::error('ID do perfil não fornecido', 400);
        }
        
        try {
            $db = Database::getUsersConnection();
            
            $fields = [];
            $params = ['id' => $data['id']];
            
            $allowedFields = ['nome', 'avatar', 'pin_protegido', 'infantil'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (isset($data['pin'])) {
                $fields[] = "pin = :pin";
                $params['pin'] = password_hash($data['pin'], PASSWORD_BCRYPT);
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            $sql = "UPDATE perfis SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Response::success(null, 'Perfil atualizado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar perfil: ' . $e->getMessage());
        }
    }
    
    private function deletePerfil() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['id'])) {
            Response::error('ID do perfil não fornecido', 400);
        }
        
        try {
            $db = Database::getUsersConnection();
            
            // Não permitir deletar perfil principal
            $sqlCheck = "SELECT tipo FROM perfis WHERE id = :id";
            $stmtCheck = $db->prepare($sqlCheck);
            $stmtCheck->execute(['id' => $data['id']]);
            $perfil = $stmtCheck->fetch();
            
            if ($perfil && $perfil['tipo'] === 'principal') {
                Response::error('Não é possível deletar o perfil principal', 400);
            }
            
            $sql = "UPDATE perfis SET ativo = 0 WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $data['id']]);
            
            Response::success(null, 'Perfil deletado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar perfil: ' . $e->getMessage());
        }
    }
    
    private function handleFavoritos($url, $method) {
        $perfilId = $_GET['perfil_id'] ?? null;
        
        if (!$perfilId) {
            Response::error('ID do perfil não fornecido', 400);
        }
        
        switch ($method) {
            case 'GET':
                $this->listFavoritos($perfilId);
                break;
                
            case 'POST':
                $this->addFavorito($perfilId);
                break;
                
            case 'DELETE':
                $this->removeFavorito($perfilId);
                break;
        }
    }
    
    private function listFavoritos($perfilId) {
        try {
            $db = Database::getUsersConnection();
            
            $sql = "SELECT * FROM favoritos WHERE perfil_id = :perfil_id ORDER BY created_at DESC";
            $stmt = $db->prepare($sql);
            $stmt->execute(['perfil_id' => $perfilId]);
            
            $favoritos = $stmt->fetchAll();
            
            Response::success($favoritos);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar favoritos: ' . $e->getMessage());
        }
    }
    
    private function addFavorito($perfilId) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'tipo_conteudo' => 'required',
            'conteudo_id' => 'required',
            'nome' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getUsersConnection();
            
            $sql = "INSERT INTO favoritos (perfil_id, tipo_conteudo, conteudo_id, nome, imagem, dados_adicionais) 
                    VALUES (:perfil_id, :tipo_conteudo, :conteudo_id, :nome, :imagem, :dados_adicionais)
                    ON DUPLICATE KEY UPDATE nome = :nome";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'perfil_id' => $perfilId,
                'tipo_conteudo' => $data['tipo_conteudo'],
                'conteudo_id' => $data['conteudo_id'],
                'nome' => $data['nome'],
                'imagem' => $data['imagem'] ?? null,
                'dados_adicionais' => isset($data['dados_adicionais']) ? json_encode($data['dados_adicionais']) : null
            ]);
            
            Response::success(null, 'Favorito adicionado com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao adicionar favorito: ' . $e->getMessage());
        }
    }
    
    private function removeFavorito($perfilId) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!isset($data['conteudo_id'])) {
            Response::error('ID do conteúdo não fornecido', 400);
        }
        
        try {
            $db = Database::getUsersConnection();
            
            $sql = "DELETE FROM favoritos WHERE perfil_id = :perfil_id AND conteudo_id = :conteudo_id";
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'perfil_id' => $perfilId,
                'conteudo_id' => $data['conteudo_id']
            ]);
            
            Response::success(null, 'Favorito removido com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao remover favorito: ' . $e->getMessage());
        }
    }
    
    private function handleHistorico($url, $method) {
        // Implementar histórico de visualização
        Response::success(['message' => 'Histórico - Em desenvolvimento']);
    }
    
    private function getRecomendacoes() {
        $perfilId = $_GET['perfil_id'] ?? null;
        
        if (!$perfilId) {
            Response::error('ID do perfil não fornecido', 400);
        }
        
        try {
            $db = Database::getUsersConnection();
            
            $sql = "SELECT * FROM recomendacoes 
                    WHERE perfil_id = :perfil_id 
                    ORDER BY score DESC 
                    LIMIT 20";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['perfil_id' => $perfilId]);
            
            $recomendacoes = $stmt->fetchAll();
            
            Response::success($recomendacoes);
            
        } catch (Exception $e) {
            Response::error('Erro ao buscar recomendações: ' . $e->getMessage());
        }
    }
    
    private function listAvatares() {
        try {
            $db = Database::getUsersConnection();
            
            $sql = "SELECT * FROM avatares WHERE ativo = 1 ORDER BY categoria, ordem";
            $stmt = $db->query($sql);
            
            $avatares = $stmt->fetchAll();
            
            Response::success($avatares);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar avatares: ' . $e->getMessage());
        }
    }
}
