<?php
/**
 * Controller de Autenticação
 */
class AuthController {
    
    public function route($url, $method) {
        $action = $url[1] ?? 'index';
        
        switch ($action) {
            case 'login':
                if ($method === 'POST') {
                    $this->login();
                }
                break;
                
            case 'refresh':
                if ($method === 'POST') {
                    $this->refresh();
                }
                break;
                
            case 'logout':
                if ($method === 'POST') {
                    $this->logout();
                }
                break;
                
            default:
                Response::notFound();
        }
    }
    
    private function login() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'email' => 'required|email',
            'senha' => 'required',
            'tipo' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        $tipo = $data['tipo']; // 'admin' ou 'provedor'
        $email = $data['email'];
        $senha = $data['senha'];
        
        try {
            $db = Database::getAdminConnection();
            
            $table = $tipo === 'admin' ? 'admins' : 'provedores';
            $sql = "SELECT * FROM {$table} WHERE email = :email AND ativo = 1";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['email' => $email]);
            $user = $stmt->fetch();
            
            if (!$user || !password_verify($senha, $user['senha'])) {
                Response::error('Credenciais inválidas', 401);
            }
            
            // Atualizar último acesso
            $updateSql = "UPDATE {$table} SET ultimo_acesso = NOW() WHERE id = :id";
            $updateStmt = $db->prepare($updateSql);
            $updateStmt->execute(['id' => $user['id']]);
            
            // Gerar tokens
            $payload = [
                'id' => $user['id'],
                'email' => $user['email'],
                'tipo' => $tipo
            ];
            
            $accessToken = JWT::encode($payload);
            $refreshToken = JWT::createRefreshToken($tipo, $user['id']);
            
            // Log
            Logger::log($tipo, $user['id'], 'login', 'Login realizado com sucesso');
            
            Response::success([
                'access_token' => $accessToken,
                'refresh_token' => $refreshToken,
                'user' => [
                    'id' => $user['id'],
                    'nome' => $user['nome'],
                    'email' => $user['email'],
                    'tipo' => $tipo
                ]
            ], 'Login realizado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao realizar login: ' . $e->getMessage());
        }
    }
    
    private function refresh() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (empty($data['refresh_token'])) {
            Response::error('Refresh token não fornecido', 400);
        }
        
        $tokenData = JWT::validateRefreshToken($data['refresh_token']);
        
        if (!$tokenData) {
            Response::error('Refresh token inválido ou expirado', 401);
        }
        
        // Gerar novo access token
        $payload = [
            'id' => $tokenData['usuario_id'],
            'tipo' => $tokenData['tipo_usuario']
        ];
        
        $accessToken = JWT::encode($payload);
        
        Response::success([
            'access_token' => $accessToken
        ], 'Token renovado com sucesso');
    }
    
    private function logout() {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (!empty($data['refresh_token'])) {
            JWT::revokeRefreshToken($data['refresh_token']);
        }
        
        Response::success(null, 'Logout realizado com sucesso');
    }
}
