<?php
/**
 * Controller do Painel Admin
 */
class AdminController {
    
    public function route($url, $method) {
        // Autenticar como admin
        $auth = AuthMiddleware::authenticateAdmin();
        
        $action = $url[1] ?? 'index';
        
        switch ($action) {
            case 'provedores':
                $this->handleProvedores($url, $method, $auth);
                break;
                
            case 'configuracoes':
                $this->handleConfiguracoes($url, $method, $auth);
                break;
                
            case 'logs':
                $this->handleLogs($url, $method, $auth);
                break;
                
            case 'financeiro':
                $this->handleFinanceiro($url, $method, $auth);
                break;
                
            default:
                Response::notFound();
        }
    }
    
    private function handleProvedores($url, $method, $auth) {
        $id = $url[2] ?? null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getProvedor($id);
                } else {
                    $this->listProvedores();
                }
                break;
                
            case 'POST':
                $this->createProvedor($auth);
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateProvedor($id, $auth);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteProvedor($id, $auth);
                }
                break;
        }
    }
    
    private function listProvedores() {
        try {
            $db = Database::getAdminConnection();
            
            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? ITEMS_PER_PAGE;
            $offset = ($page - 1) * $limit;
            
            $sql = "SELECT id, numero_provedor, nome, email, logo, ativo, data_vencimento, created_at 
                    FROM provedores 
                    ORDER BY created_at DESC 
                    LIMIT :limit OFFSET :offset";
            
            $stmt = $db->prepare($sql);
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $provedores = $stmt->fetchAll();
            
            // Contar total
            $countSql = "SELECT COUNT(*) as total FROM provedores";
            $total = $db->query($countSql)->fetch()['total'];
            
            Response::success([
                'provedores' => $provedores,
                'pagination' => [
                    'page' => (int)$page,
                    'limit' => (int)$limit,
                    'total' => (int)$total,
                    'pages' => ceil($total / $limit)
                ]
            ]);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar provedores: ' . $e->getMessage());
        }
    }
    
    private function getProvedor($id) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM provedores WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            $provedor = $stmt->fetch();
            
            if (!$provedor) {
                Response::notFound('Provedor não encontrado');
            }
            
            Response::success($provedor);
            
        } catch (Exception $e) {
            Response::error('Erro ao buscar provedor: ' . $e->getMessage());
        }
    }
    
    private function createProvedor($auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'numero_provedor' => 'required',
            'nome' => 'required',
            'email' => 'required|email',
            'senha' => 'required|min:6',
            'url_principal' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getAdminConnection();
            
            // Verificar se número do provedor já existe
            $checkSql = "SELECT id FROM provedores WHERE numero_provedor = :numero_provedor";
            $checkStmt = $db->prepare($checkSql);
            $checkStmt->execute(['numero_provedor' => $data['numero_provedor']]);
            
            if ($checkStmt->fetch()) {
                Response::error('Número de provedor já cadastrado', 400);
            }
            
            $sql = "INSERT INTO provedores (numero_provedor, nome, email, senha, url_principal, url_backup_1, url_backup_2) 
                    VALUES (:numero_provedor, :nome, :email, :senha, :url_principal, :url_backup_1, :url_backup_2)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'numero_provedor' => $data['numero_provedor'],
                'nome' => $data['nome'],
                'email' => $data['email'],
                'senha' => password_hash($data['senha'], PASSWORD_BCRYPT),
                'url_principal' => $data['url_principal'],
                'url_backup_1' => $data['url_backup_1'] ?? null,
                'url_backup_2' => $data['url_backup_2'] ?? null
            ]);
            
            $provedorId = $db->lastInsertId();
            
            Logger::log('admin', $auth['id'], 'criar_provedor', "Provedor {$data['nome']} criado", ['provedor_id' => $provedorId]);
            
            Response::success(['id' => $provedorId], 'Provedor criado com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar provedor: ' . $e->getMessage());
        }
    }
    
    private function updateProvedor($id, $auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            $db = Database::getAdminConnection();
            
            $fields = [];
            $params = ['id' => $id];
            
            $allowedFields = ['nome', 'email', 'url_principal', 'url_backup_1', 'url_backup_2', 'ativo', 'data_vencimento'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (isset($data['senha'])) {
                $fields[] = "senha = :senha";
                $params['senha'] = password_hash($data['senha'], PASSWORD_BCRYPT);
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            $sql = "UPDATE provedores SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Logger::log('admin', $auth['id'], 'atualizar_provedor', "Provedor ID {$id} atualizado");
            
            Response::success(null, 'Provedor atualizado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar provedor: ' . $e->getMessage());
        }
    }
    
    private function deleteProvedor($id, $auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "DELETE FROM provedores WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            Logger::log('admin', $auth['id'], 'deletar_provedor', "Provedor ID {$id} deletado");
            
            Response::success(null, 'Provedor deletado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar provedor: ' . $e->getMessage());
        }
    }
    
    private function handleConfiguracoes($url, $method, $auth) {
        $id = $url[2] ?? null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getConfiguracao($id);
                } else {
                    $this->listConfiguracoes();
                }
                break;
                
            case 'POST':
                $this->createConfiguracao($auth);
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updateConfiguracao($id, $auth);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deleteConfiguracao($id, $auth);
                }
                break;
        }
    }
    
    private function listConfiguracoes() {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM configuracoes_gerais ORDER BY chave";
            $stmt = $db->query($sql);
            
            $configuracoes = $stmt->fetchAll();
            
            Response::success($configuracoes);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar configurações: ' . $e->getMessage());
        }
    }
    
    private function getConfiguracao($id) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT * FROM configuracoes_gerais WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            $config = $stmt->fetch();
            
            if (!$config) {
                Response::notFound('Configuração não encontrada');
            }
            
            Response::success($config);
            
        } catch (Exception $e) {
            Response::error('Erro ao buscar configuração: ' . $e->getMessage());
        }
    }
    
    private function createConfiguracao($auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'chave' => 'required',
            'valor' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getAdminConnection();
            
            $sql = "INSERT INTO configuracoes_gerais (chave, valor, descricao, tipo) 
                    VALUES (:chave, :valor, :descricao, :tipo)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'chave' => $data['chave'],
                'valor' => $data['valor'],
                'descricao' => $data['descricao'] ?? null,
                'tipo' => $data['tipo'] ?? 'string'
            ]);
            
            $configId = $db->lastInsertId();
            
            Logger::log('admin', $auth['id'], 'criar_configuracao', "Configuração {$data['chave']} criada");
            
            Response::success(['id' => $configId], 'Configuração criada com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar configuração: ' . $e->getMessage());
        }
    }
    
    private function updateConfiguracao($id, $auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            $db = Database::getAdminConnection();
            
            $fields = [];
            $params = ['id' => $id];
            
            $allowedFields = ['chave', 'valor', 'descricao', 'tipo'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            $sql = "UPDATE configuracoes_gerais SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Logger::log('admin', $auth['id'], 'atualizar_configuracao', "Configuração ID {$id} atualizada");
            
            Response::success(null, 'Configuração atualizada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar configuração: ' . $e->getMessage());
        }
    }
    
    private function deleteConfiguracao($id, $auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "DELETE FROM configuracoes_gerais WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            Logger::log('admin', $auth['id'], 'deletar_configuracao', "Configuração ID {$id} deletada");
            
            Response::success(null, 'Configuração deletada com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar configuração: ' . $e->getMessage());
        }
    }
    
    private function handleLogs($url, $method, $auth) {
        if ($method === 'GET') {
            $this->listLogs();
        }
    }
    
    private function listLogs() {
        try {
            $db = Database::getAdminConnection();
            
            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? 50;
            $offset = ($page - 1) * $limit;
            
            // Filtros opcionais
            $tipoUsuario = $_GET['tipo_usuario'] ?? null;
            $usuarioId = $_GET['usuario_id'] ?? null;
            $acao = $_GET['acao'] ?? null;
            $dataInicio = $_GET['data_inicio'] ?? null;
            $dataFim = $_GET['data_fim'] ?? null;
            
            $where = [];
            $params = [];
            
            if ($tipoUsuario) {
                $where[] = "tipo_usuario = :tipo_usuario";
                $params['tipo_usuario'] = $tipoUsuario;
            }
            
            if ($usuarioId) {
                $where[] = "usuario_id = :usuario_id";
                $params['usuario_id'] = $usuarioId;
            }
            
            if ($acao) {
                $where[] = "acao LIKE :acao";
                $params['acao'] = "%{$acao}%";
            }
            
            if ($dataInicio) {
                $where[] = "created_at >= :data_inicio";
                $params['data_inicio'] = $dataInicio;
            }
            
            if ($dataFim) {
                $where[] = "created_at <= :data_fim";
                $params['data_fim'] = $dataFim . ' 23:59:59';
            }
            
            $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
            
            $sql = "SELECT * FROM logs_sistema 
                    {$whereClause}
                    ORDER BY created_at DESC 
                    LIMIT :limit OFFSET :offset";
            
            $stmt = $db->prepare($sql);
            
            foreach ($params as $key => $value) {
                $stmt->bindValue(":{$key}", $value);
            }
            
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $logs = $stmt->fetchAll();
            
            // Contar total
            $countSql = "SELECT COUNT(*) as total FROM logs_sistema {$whereClause}";
            $countStmt = $db->prepare($countSql);
            
            foreach ($params as $key => $value) {
                $countStmt->bindValue(":{$key}", $value);
            }
            
            $countStmt->execute();
            $total = $countStmt->fetch()['total'];
            
            Response::success([
                'logs' => $logs,
                'pagination' => [
                    'page' => (int)$page,
                    'limit' => (int)$limit,
                    'total' => (int)$total,
                    'pages' => ceil($total / $limit)
                ]
            ]);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar logs: ' . $e->getMessage());
        }
    }
    
    private function handleFinanceiro($url, $method, $auth) {
        $id = $url[2] ?? null;
        
        switch ($method) {
            case 'GET':
                if ($id) {
                    $this->getPagamento($id);
                } else {
                    $this->listPagamentos();
                }
                break;
                
            case 'POST':
                $this->createPagamento($auth);
                break;
                
            case 'PUT':
                if ($id) {
                    $this->updatePagamento($id, $auth);
                }
                break;
                
            case 'DELETE':
                if ($id) {
                    $this->deletePagamento($id, $auth);
                }
                break;
        }
    }
    
    private function listPagamentos() {
        try {
            $db = Database::getAdminConnection();
            
            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? ITEMS_PER_PAGE;
            $offset = ($page - 1) * $limit;
            
            // Filtros opcionais
            $provedorId = $_GET['provedor_id'] ?? null;
            $status = $_GET['status'] ?? null;
            $mesAno = $_GET['mes_ano'] ?? null;
            
            $where = [];
            $params = [];
            
            if ($provedorId) {
                $where[] = "cf.provedor_id = :provedor_id";
                $params['provedor_id'] = $provedorId;
            }
            
            if ($status) {
                $where[] = "cf.status = :status";
                $params['status'] = $status;
            }
            
            if ($mesAno) {
                $where[] = "DATE_FORMAT(cf.data_vencimento, '%Y-%m') = :mes_ano";
                $params['mes_ano'] = $mesAno;
            }
            
            $whereClause = !empty($where) ? 'WHERE ' . implode(' AND ', $where) : '';
            
            $sql = "SELECT cf.*, p.nome as provedor_nome, p.numero_provedor 
                    FROM controle_financeiro cf
                    INNER JOIN provedores p ON cf.provedor_id = p.id
                    {$whereClause}
                    ORDER BY cf.data_vencimento DESC 
                    LIMIT :limit OFFSET :offset";
            
            $stmt = $db->prepare($sql);
            
            foreach ($params as $key => $value) {
                $stmt->bindValue(":{$key}", $value);
            }
            
            $stmt->bindValue(':limit', (int)$limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', (int)$offset, PDO::PARAM_INT);
            $stmt->execute();
            
            $pagamentos = $stmt->fetchAll();
            
            // Contar total
            $countSql = "SELECT COUNT(*) as total FROM controle_financeiro cf {$whereClause}";
            $countStmt = $db->prepare($countSql);
            
            foreach ($params as $key => $value) {
                $countStmt->bindValue(":{$key}", $value);
            }
            
            $countStmt->execute();
            $total = $countStmt->fetch()['total'];
            
            // Estatísticas
            $statsSql = "SELECT 
                            SUM(CASE WHEN status = 'pago' THEN valor ELSE 0 END) as total_pago,
                            SUM(CASE WHEN status = 'pendente' THEN valor ELSE 0 END) as total_pendente,
                            SUM(CASE WHEN status = 'atrasado' THEN valor ELSE 0 END) as total_atrasado
                         FROM controle_financeiro cf {$whereClause}";
            
            $statsStmt = $db->prepare($statsSql);
            
            foreach ($params as $key => $value) {
                $statsStmt->bindValue(":{$key}", $value);
            }
            
            $statsStmt->execute();
            $stats = $statsStmt->fetch();
            
            Response::success([
                'pagamentos' => $pagamentos,
                'estatisticas' => $stats,
                'pagination' => [
                    'page' => (int)$page,
                    'limit' => (int)$limit,
                    'total' => (int)$total,
                    'pages' => ceil($total / $limit)
                ]
            ]);
            
        } catch (Exception $e) {
            Response::error('Erro ao listar pagamentos: ' . $e->getMessage());
        }
    }
    
    private function getPagamento($id) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "SELECT cf.*, p.nome as provedor_nome, p.numero_provedor 
                    FROM controle_financeiro cf
                    INNER JOIN provedores p ON cf.provedor_id = p.id
                    WHERE cf.id = :id";
            
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            $pagamento = $stmt->fetch();
            
            if (!$pagamento) {
                Response::notFound('Pagamento não encontrado');
            }
            
            Response::success($pagamento);
            
        } catch (Exception $e) {
            Response::error('Erro ao buscar pagamento: ' . $e->getMessage());
        }
    }
    
    private function createPagamento($auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        $validator = new Validator();
        if (!$validator->validate($data, [
            'provedor_id' => 'required|numeric',
            'valor' => 'required|numeric',
            'data_vencimento' => 'required'
        ])) {
            Response::error('Dados inválidos', 400, $validator->getErrors());
        }
        
        try {
            $db = Database::getAdminConnection();
            
            $sql = "INSERT INTO controle_financeiro (provedor_id, valor, data_vencimento, status, observacoes) 
                    VALUES (:provedor_id, :valor, :data_vencimento, :status, :observacoes)";
            
            $stmt = $db->prepare($sql);
            $stmt->execute([
                'provedor_id' => $data['provedor_id'],
                'valor' => $data['valor'],
                'data_vencimento' => $data['data_vencimento'],
                'status' => $data['status'] ?? 'pendente',
                'observacoes' => $data['observacoes'] ?? null
            ]);
            
            $pagamentoId = $db->lastInsertId();
            
            Logger::log('admin', $auth['id'], 'criar_pagamento', "Pagamento criado para provedor ID {$data['provedor_id']}");
            
            Response::success(['id' => $pagamentoId], 'Pagamento criado com sucesso', 201);
            
        } catch (Exception $e) {
            Response::error('Erro ao criar pagamento: ' . $e->getMessage());
        }
    }
    
    private function updatePagamento($id, $auth) {
        $data = json_decode(file_get_contents('php://input'), true);
        
        try {
            $db = Database::getAdminConnection();
            
            $fields = [];
            $params = ['id' => $id];
            
            $allowedFields = ['valor', 'data_vencimento', 'data_pagamento', 'status', 'metodo_pagamento', 'comprovante', 'observacoes'];
            
            foreach ($allowedFields as $field) {
                if (isset($data[$field])) {
                    $fields[] = "{$field} = :{$field}";
                    $params[$field] = $data[$field];
                }
            }
            
            if (empty($fields)) {
                Response::error('Nenhum campo para atualizar', 400);
            }
            
            // Se status mudou para 'pago' e não tem data_pagamento, adicionar data atual
            if (isset($data['status']) && $data['status'] === 'pago' && !isset($data['data_pagamento'])) {
                $fields[] = "data_pagamento = NOW()";
            }
            
            $sql = "UPDATE controle_financeiro SET " . implode(', ', $fields) . " WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute($params);
            
            Logger::log('admin', $auth['id'], 'atualizar_pagamento', "Pagamento ID {$id} atualizado");
            
            Response::success(null, 'Pagamento atualizado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao atualizar pagamento: ' . $e->getMessage());
        }
    }
    
    private function deletePagamento($id, $auth) {
        try {
            $db = Database::getAdminConnection();
            
            $sql = "DELETE FROM controle_financeiro WHERE id = :id";
            $stmt = $db->prepare($sql);
            $stmt->execute(['id' => $id]);
            
            Logger::log('admin', $auth['id'], 'deletar_pagamento', "Pagamento ID {$id} deletado");
            
            Response::success(null, 'Pagamento deletado com sucesso');
            
        } catch (Exception $e) {
            Response::error('Erro ao deletar pagamento: ' . $e->getMessage());
        }
    }
}
